<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_Travel_Cart_Checkout_Addon' ) ) :

	/**
	 * Main WP_Travel_Cart_Checkout_Addon Class (singleton).
	 *
	 * @since 1.0.0
	 */
	final class WP_Travel_Cart_Checkout_Addon {

		/**
		 * Plugin Name.
		 *
		 * @var string
		 */
		public $plugin_name = 'wp-travel-cart';

		/**
		 * Assets Path.
		 *
		 * @var string
		 */
		public $assets_path;

		/**
		 * WP Travel Stripe version.
		 *
		 * @var string
		 */
		public $version = '1.0.0';
		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Admin Settings Page.
		 *
		 * @var string
		 */
		static $collection = 'settings';

		/**
		 * Main WP_Travel_Utilities Instance.
		 * Ensures only one instance of WP_Travel_Utilities is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Utilities()
		 * @return WP_Travel_Utilities - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * WP_Travel_Utilities Constructor.
		 */
		public function __construct() {

			if ( ! session_id() && ! headers_sent() ) {
				session_start();
				session_write_close();
			}

			$this->define_constants();
			$this->assets_path = plugin_dir_url( WP_TRAVEL_CART_CHECKOUT_PLUGIN_FILE ) . 'assets/';
			$this->includes();
			$this->init_hooks();

		}

		/**
		 * Define Constants.
		 */
		private function define_constants() {
			$this->define( 'WP_TRAVEL_CART_CHECKOUT_PLUGIN_FILE', __FILE__ );
			$this->define( 'WP_TRAVEL_CART_CHECKOUT_ABSPATH', dirname( __FILE__ ) . '/' );
			$this->define( 'WP_TRAVEL_CART_CHECKOUT_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
			$this->define( 'WP_TRAVEL_CART_CHECKOUT_PLUGIN_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) );
			$this->define( 'WP_TRAVEL_CART_CHECKOUT_VERSION', $this->version );
			$this->define( 'WP_TRAVEL_CART_CHECKOUT_PLUGIN_NAME', __( 'WP Travel Cart Checkout Addon', 'wp-travel-pro' ) );
		}

		/**
		 * Hook into actions and filters.
		 *
		 * @since 1.0.0
		 * @return void
		 */
		private function init_hooks() {

			add_filter( 'wp_travel_is_force_checkout_enabled', array( $this, 'wp_travel_utils_is_force_checkout_enabled' ) );

			add_filter( 'wp_travel_checkout_fields', array( $this, 'checkout_form_fields_sort' ) );

			add_filter( 'wptravel_frontend_data', array( $this, 'frontend_vars' ) );

			// Allow Multiple Cart / Checkout process.
			add_filter( 'wptravel_enable_multiple_checkout', 'wp_travel_utilities_is_multiple_cart_enabled' );

		}

		/**
		 * Define constant if not already set.
		 *
		 * @param  string $name  Name of constant.
		 * @param  string $value Value of constant.
		 * @return void
		 */
		private function define( $name, $value ) {
			if ( ! defined( $name ) ) {
				define( $name, $value );
			}
		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public function includes() {

			if ( class_exists( 'WP_Travel' ) ) {
				/**
				 * Include required files
				 */
				// Functions.
				// include sprintf( '%s/inc/functions.php', WP_TRAVEL_CART_CHECKOUT_ABSPATH );

				if ( $this->is_request( 'admin' ) ) :

					include sprintf( '%s/admin/wp-travel-cart-checkout-additional-settings.php', WP_TRAVEL_CART_CHECKOUT_ABSPATH );

					include sprintf( '%s/admin/class-backend-additional-meta-box.php', WP_TRAVEL_CART_CHECKOUT_ABSPATH );

				endif;
			}
		}

		/**
		 * Frontend Vars.
		 *
		 * @param array $vars array.
		 * @return array $vars Modified Array.
		 */
		public function frontend_vars( $vars ) {

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$skip_cart_page_booking = isset( $settings['skip_cart_page_booking'] ) && ! empty( $settings['skip_cart_page_booking'] ) ? $settings['skip_cart_page_booking'] : 'no';

			if ( wp_travel_utilities_is_multiple_cart_enabled() ) {

				global $post;

				if ( $post ) {
					$post_id = $post->ID;

					$vars['cartUrl'] = add_query_arg( 'add_to_cart', 'true', get_the_permalink( $post_id ) );
				}
			}

			if ( is_array( $vars ) && 'yes' === $skip_cart_page_booking ) {

				$vars['cartUrl'] = wptravel_get_page_permalink( 'wp-travel-checkout' );

			}

			return $vars;

		}

		/**
		 * This will uninstall this plugin if parent WP-Travel plugin not found
		 */
		public function wp_travel_check_dependency() {

			$plugin      = plugin_basename( __FILE__ );
			$plugin_data = get_plugin_data( __FILE__, false );

			if ( ! class_exists( 'WP_Travel' ) ) {

				if ( is_plugin_active( $plugin ) ) {
					deactivate_plugins( $plugin );
					wp_die( wp_kses_post( '<strong>' . $plugin_data['Name'] . '</strong> requires the WP Travel plugin to work. Please activate it first. <br /><br />Back to the WordPress <a href="' . esc_url( get_admin_url( null, 'plugins.php' ) ) . '">Plugins page</a>.' ) );
				}
			}
		}
		/**
		 * Booking Path Override
		 */
		public function wp_travel_utils_is_force_checkout_enabled() {

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$force_checkout = isset( $settings['force_cart_checkout_process_for_trips'] ) && ! empty( $settings['force_cart_checkout_process_for_trips'] ) ? $settings['force_cart_checkout_process_for_trips'] : 'no';

			if ( 'yes' === $force_checkout ) {

				return true;
			}

			return false;

		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}
		/**
		 * Modify the checkout form fields.
		 */
		public function checkout_form_fields_sort( $fields ) {

			global $wt_cart;

			$cart_amounts = $wt_cart->get_total();

			$cart_items = $wt_cart->getItems();

			$cart_trip = '';

			if ( ! empty( $cart_items ) && is_array( $cart_items ) ) {

				$cart_trip = array_slice( $cart_items, 0, 1 );
				$cart_trip = array_shift( $cart_trip );

			}
			$trip_id    = isset( $cart_trip['trip_id'] ) ? $cart_trip['trip_id'] : '';
			$trip_price = isset( $cart_trip['trip_price'] ) && ! empty( $cart_trip['trip_price'] ) ? $cart_trip['trip_price'] : '';

			$checkout_info_label  = get_post_meta( $trip_id, 'wp_travel_trip_checkout_page_info_label', true );
			$additional_data_info = get_post_meta( $trip_id, 'wp_travel_trip_checkout_page_info', true );

			if ( ! empty( $additional_data_info ) ) {

				$fields['payment_fields']['checkout_trip_information'] = array(
					'type'         => 'text_info',
					'label'        => $checkout_info_label,
					'before_field' => '',
					'name'         => 'wp_travel_checkout_trip_information',
					'id'           => 'wp-travel-option-checkout-trip-info',
					'validations'  => array(
						'required' => true,
					),
					'default'      => $additional_data_info,
					'priority'     => 100,
				);

			}

			$trip_pricing_options_data = get_post_meta( $trip_id, 'wp_travel_pricing_options', true );

			if ( method_exists( 'WP_Travel',  'is_page' ) ) { // @since WP Travel 4.4.2
				$is_checkout_page = WP_Travel::is_page( 'checkout' );
			} else {
				$is_checkout_page = wp_travel_is_checkout_page();
			}

			if ( $is_checkout_page && ( empty( $trip_price ) || ! wptravel_is_payment_enabled() || ( is_array( $trip_pricing_options_data ) && count( $trip_pricing_options_data ) === 0 ) ) ) {

				// unset( $fields['payment_fields']['booking_option'] );
				unset( $fields['payment_fields']['payment_gateway'] );
				unset( $fields['payment_fields']['trip_price'] );
				unset( $fields['payment_fields']['payment_mode'] );
				unset( $fields['payment_fields']['trip_price_info'] );
				unset( $fields['payment_fields']['payment_amount_info'] );

			}

			return $fields;
		}
	}

endif;
/**
 * Main instance of WP Travel Cart Checkout Addon.
 *
 * Returns the main instance of WP_travel_stripe to prevent the need to use globals.
 *
 * @since  1.0.0
 * @return WP Travel Cart Checkout Addon
 */
function wp_travel_cart_checkout_addon() {
	return WP_Travel_Cart_Checkout_Addon::instance();
}

// Start WP Travel Utilities Addon.
wp_travel_cart_checkout_addon();
