<?php 
/**
 * Register custom rest routes
 * endpoints looks like http://example.com/wp-json/wp-travel/v1/travel-guide
 *
 * @see https://developer.wordpress.org/rest-api/extending-the-rest-api/adding-custom-endpoints/
 */
namespace WP_Travel_Travel_Guide_Core;
use WP_REST_Request;

if( !class_exists( 'Api' ) ){

	class Api extends Base_Api{

		/**
		 * The single instance of the class.
		 *
		 * @var string $_instance
		 */
		protected static $instance = null;

		/**
		 * Database.
		 *
		 */
		protected $db = null;


		/**
		 * Main WP_Travel_travel_guide Instance.
		 * Ensures only one instance of WP_Travel_travel_guide is loaded or can be loaded.
		 *
		 * @return WP_Travel_travel_guide - Main instance.
		 */
		public static function get_instance() {
		    if ( ! self::$instance ) {
		        self::$instance = new self();
		    }
		    return self::$instance;
		}


		/**
		 * Constructor.
		 */
		public function __construct(){
			parent::__construct();
			$this->db = new Database();
		}


		/**
		 * Defining routes.
		 */
		public function get_routes(){

			$routes = array(
				array(
					'route'    => '',
					'method'   => 'get',
					'callback' => array( $this, 'get' ) # get all the Travel Guides
				),

				array(
					'route'    => '(?P<id>\d+)',
					'method'   => 'get',
					'callback' => array( $this, 'get_by_id' ), # get Travel Guide by their id
					'args'     => array(
						'id' => array(
							'validate_callback' => array( $this, 'is_numeric' )
						)
				    )
				),

				array(
					'route'    => '',
					'method'   => 'post',
					'callback' => array( $this, 'save' ) # create new Travel Guide
				),

				array(
					'route'    => '(?P<id>\d+)',
					'method'   => 'put',
					'callback' => array( $this, 'update' ),
					'args'     => array(
						'id' => array(
							'validate_callback' => array( $this, 'is_numeric' )
						)
				    )
				),
				array(
					'route'    => '(?P<id>\d+)',
					'method'   => 'delete',
					'callback' => array( $this, 'delete' ),
					'args'     => array(
						'id' => array(
							'validate_callback' => array( $this, 'is_numeric' )
						)
				    )
				)
			);

			return $routes;
		}


		/**
		 * Defining delete operation route callback.
		 */
		public function delete( WP_REST_Request $request ){
			$id = $request->get_param( 'id' );

			// Delete data from Travel Guide post type
			wp_delete_post( get_page_by_path( get_userdata( $id )->data->user_login, OBJECT, 'travel-guide' )->ID );

			$deleted = $this->db->delete( $id );
			if( $deleted ){
				$this->set_data(array(
					'deleted_id' => $id
				));

				$this->set_message( esc_html__( 'Travel guide deleted successfully.', 'wp-travel-pro' ) );
			}else{

				$this->set_message( esc_html__( 'Travel guide not deleted.', 'wp-travel-pro' ) );
			}

			return $this->get_response();
		}

		/**
		 * Defining route callback to get all guide data.
		 */
		public function get( WP_REST_Request $request ){

			$role = Role::get_instance();
			$args = array(
			    'role'    => $role->get(),
			    'orderby' => 'ID',
			    'order'   => 'DESC'
			);

			$users = $this->db->get( $args );

			if( count( $users ) > 0 ){
				$this->set_data( $users );
				$this->set_message( esc_html__( 'Travel guide fetched successfully.', 'wp-travel-pro' ) );
				
			}else{
				$this->set_message( esc_html__( 'Travel guides not found', 'wp-travel-pro' ) );
				$this->set_status( 400 );
			}

			return $this->get_response();
		}

		/**
		 * Defining route callback to get all guide data by id.
		 */
		public function get_by_id( WP_REST_Request $request ){
			$id   = $request->get_param( 'id' );

			$user = $this->db->get( array( 'search' => $id, 'search_columns' => ['ID'] ) );
			if( $user ){
				$this->set_message( esc_html__( 'Travel Guide fetched successfully.', 'wp-travel-pro' ) );
				$this->set_data( $user[0] );
			}else{
				$this->set_status( 400 );
				$this->set_message( esc_html__( 'Travel Guide not found.', 'wp-travel-pro' ) );
			}

			return $this->get_response();
		}

		/**
		 * Defining route callback to save guide data.
		 */
		public function save( WP_REST_Request $request ){

			$data = $request->get_params();

			// Added post to post type Travel Guide
			$guide_post_arg = array(
			  'post_title'    => $data[ 'user_login' ],
			  'post_content'  => '',
			  'post_status'   => 'publish',
			  'post_type'     => 'travel-guide',
			);
			
			$guide_id = wp_insert_post( $guide_post_arg );

			if( $guide_id && isset($data['profile_picture']['id']) ){
				update_post_meta( $guide_id, '_thumbnail_id', $data['profile_picture']['id'] );
			}
			// var_dump( $guide_id  );
			// die;

			$errors = array();

			if( empty( $data[ 'user_login' ] ) ){
				$errors[ 'user_login' ] = esc_html__( 'Username cannot be empty.', 'wp-travel-pro' );
			}else if( username_exists( $data[ 'user_login' ] ) ){
				$errors[ 'user_login' ] = esc_html__( 'Username already exists', 'wp-travel-pro' );
			}

			if( empty( $data[ 'user_email' ] ) ){
				$errors[ 'user_email' ] = esc_html__( 'Email address cannot be empty.', 'wp-travel-pro' );
			}else{

				if( !is_email( $data[ 'user_email' ] ) ){
					$errors[ 'user_email' ] = esc_html__( 'Invalid user email.', 'wp-travel-pro' );
				}elseif( email_exists( $data[ 'user_email' ] ) ){
					$errors[ 'user_email' ] = esc_html__( 'Sorry, that email address is already used!', 'wp-travel-pro' );
				}
			}

			if( count( $errors ) > 0 ){

				$this->set_validation_errors( $errors );
				$this->set_message( esc_html__( 'Bad Request', 'wp-travel-pro' ) );
				$this->set_status( 400 );

			}else{

				$role = Role::get_instance();

				$this->new_user_data = $data;
				$user_id = $this->db->save(array(
					'user_login'   => wp_slash( $data[ 'user_login' ] ),
					'user_email'   => wp_slash( $data[ 'user_email' ] ),
					'user_pass'    => wp_generate_password(),
					'role'         => $role->get(),
					'user_url'     => $this->get_meta( 'user_url' ),
					'nickname'     => $this->get_meta( 'nickname' ),
					'first_name'   => $this->get_meta( 'first_name' ),
					'last_name'    => $this->get_meta( 'last_name' ),
					'display_name' => $this->get_meta( 'display_name' ),
					'description'  => $this->get_meta( 'description' ),
					'meta_input'   => array(
						'guide_unique_id'=> $this->get_meta( 'guide_unique_id' ),
						'user_slogan'  	=> $this->get_meta( 'user_slogan' ),
						'user_phone'  	=> $this->get_meta( 'user_phone' ),
						'city'        	=> $this->get_meta( 'city' ),
						'country'     	=> $this->get_meta( 'country' ),
						'age'         	=> $this->get_meta( 'age' ),
						'join_year'     => $this->get_meta( 'join_year' ),
						'facebook_link' => $this->get_meta( 'facebook_link' ),
						'instagram_link' => $this->get_meta( 'instagram_link' ),
						'twitter_link' => $this->get_meta( 'twitter_link' ),
						'linkedin_link' => $this->get_meta( 'linkedin_link' ),
						'youtube_link' => $this->get_meta( 'youtube_link' ),
						'tiktok_link' => $this->get_meta( 'tiktok_link' ),
						'profile_picture' => $this->get_meta( 'profile_picture' ),
						'language'    	=> $this->get_meta( 'language' ),
						'gender'    	=> $this->get_meta( 'gender' ),
						'description' 	=> $this->get_meta( 'description' ),
						'biography' 	=> $this->get_meta( 'biography' ),
						'trip_list' 	=> $this->get_meta( 'trip_list' ),
						'profile_picture' 	=> $this->get_meta( 'profile_picture' ),
					)
				));

				if( $user_id ){
					$this->set_message( esc_html__( 'Travel Guide added successfully.', 'wp-travel-pro' ) );
					$this->set_data( array( 'id' => $user_id ) );
				}else{
					$this->set_message( esc_html__( 'Something went wrong', 'wp-travel-pro' ) );
					$this->set_status( 500 );
				}
			}

			return $this->get_response();



		}

		public function get_meta( $key, $default = '' ){
			return !empty( $this->new_user_data[ $key ] ) ? $this->new_user_data[ $key ] : $default;
		}


		/**
		 * Defining route callback to update guide data.
		 */
		public function update( WP_REST_Request $request ){

			$data = $request->get_params();


			if( !empty( $data['profile_picture'] ) ){
				update_post_meta( 
					get_posts(
						array(
							'post_type'              => 'travel-guide',
							'title'                  => $data['user_login'],
						)
					)[0]->ID, 
					'_thumbnail_id', 
					$data['profile_picture']['id'] 
				);
			}
			

			$this->new_user_data = $data;
			$user_id = $this->db->update( $data[ 'id' ], array(
				'user_email'   => wp_slash( $this->get_meta( 'user_email' ) ),
				'user_pass'    => $this->get_meta( 'user_pass' ),
				'user_url'     => $this->get_meta( 'user_url' ),
				'nickname'     => $this->get_meta( 'nickname' ),
				'first_name'   => $this->get_meta( 'first_name' ),
				'last_name'    => $this->get_meta( 'last_name' ),
				'display_name' => $this->get_meta( 'display_name' ),
				'description'  => $this->get_meta( 'description' ),
				'meta_input'   => array(
					'user_slogan'  	=> $this->get_meta( 'user_slogan' ),
					'user_phone'  	=> $this->get_meta( 'user_phone' ),
					'city'        	=> $this->get_meta( 'city' ),
					'country'     	=> $this->get_meta( 'country' ),
					'age'         	=> $this->get_meta( 'age' ),
					'join_year'     => $this->get_meta( 'join_year' ),
					'facebook_link' => $this->get_meta( 'facebook_link' ),
					'instagram_link' => $this->get_meta( 'instagram_link' ),
					'twitter_link' => $this->get_meta( 'twitter_link' ),
					'linkedin_link' => $this->get_meta( 'linkedin_link' ),
					'youtube_link' => $this->get_meta( 'youtube_link' ),
					'tiktok_link' => $this->get_meta( 'tiktok_link' ),
					'language'    	=> $this->get_meta( 'language' ),
					'gender'    	=> $this->get_meta( 'gender' ),
					'description' 	=> $this->get_meta( 'description' ),
					'biography' 	=> $this->get_meta( 'biography' ),
					'trip_list' 	=> $this->get_meta( 'trip_list' ),
					'profile_picture' 	=> $this->get_meta( 'profile_picture' ),
				)
			));

			if( is_wp_error( $user_id ) ){
				$this->set_status( 400 );
				$this->set_message( esc_html__( 'Travel Guide update failed.', 'wp-travel-pro' ) );
			}else if( $user_id ){
				$this->set_data( $data );
				$this->set_message( esc_html__( 'Travel Guide update successfully.', 'wp-travel-pro' ) );
			}

			return $this->get_response();
		}
	}

	Api::get_instance();

}
