<?php
/**
 * WP Travel Tour Extras Core Class.
 *
 * @package wp-travel-tour-extras-core
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Tour Extras core.
if ( ! class_exists( 'WP_Travel_Tour_Extras_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Tour_Extras_Core {
		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Tour Extras Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Tour_Extras_Core Instance.
		 * Ensures only one instance of WP_Travel_Tour_Extras_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Tour_Extras_Core()
		 * @return WP_Travel_Tour_Extras_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];
			add_filter( 'wp_travel_extras_is_pro_enabled', '__return_true' );

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$enable_tour_extras = isset( $settings['show_wp_travel_tour_extras'] ) ? $settings['show_wp_travel_tour_extras'] : 'yes';

			add_filter( 'wp_travel_settings_fields', array( 'WP_Travel_Tour_Extras_Core', 'settings_fields' ) );
			add_action( 'wp_travel_addons_setings_tab_fields', array( 'WP_Travel_Tour_Extras_Core', 'plugin_action' ) );
			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Tour_Extras_Core', 'admin_assets' ) );
			// if ( 'yes' !== $enable_tour_extras ) {
			// 	add_action( 'wp_travel_extras_pro_single_options', array( 'WP_Travel_Tour_Extras_Core', 'plugin_disabled_notice' ) );
			// 	return;
			// }

			self::includes();

			if ( ! session_id() && ! headers_sent() ) {
				session_start();
				session_write_close();
			}
			add_filter( 'register_post_type_args', array( 'WP_Travel_Tour_Extras_Core', 'register_tour_extras' ), 1, 2 );

			// add_action( 'init', array( 'WP_Travel_Tour_Extras_Core', 'includes' ) );
			add_action( 'wp_enqueue_scripts', array( 'WP_Travel_Tour_Extras_Core', 'frontend_assets' ) );

			// Trip extra fields in trip dates and price tab. @since 2.0.3
			add_action( 'wp_travel_trip_extras_fields', array( 'WP_Travel_Tour_Extras_Core', 'trip_extra_fields' ), 10, 4 );
			add_action( 'wp_travel_itinerary_extra_meta_save', array( 'WP_Travel_Tour_Extras_Core', 'save_meta_data' ) );

			/**
			 * Filter
			 *
			 * @since wptravel 4.0.6
			 */
			add_filter(
				'wp-travel-script-handles',
				function( $handles ) {
					$handles[] = 'wp-travel-tour-extras-front-end-js';
					$handles[] = 'wp-travel-tour-extras-cart-js';
					return $handles;
				}
			);
		}

		/**
		 * Default settings fields.
		 *
		 * @param array $settings WP Travel Settings.
		 * @return array
		 */
		public static function settings_fields( $settings ) {
			$settings['show_wp_travel_tour_extras'] = 'yes';
			return $settings;
		}

		/**
		 * Plugin action to show / hide plugin settings and features.
		 */
		public static function plugin_action() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$field_name  = 'show_wp_travel_tour_extras';
			$field_label = __( 'WP Travel Tour Extras', 'wp-travel-pro' );

			$field_value = isset( $settings[ $field_name ] ) ? $settings[ $field_name ] : 'yes'; ?>
			<table class="form-table">
				<tr>
					<th>
						<label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label>
					</th>
					<td>
						<span class="show-in-frontend checkbox-default-design">
							<label data-on="ON" data-off="OFF">
								<input value="no" name="<?php echo esc_attr( $field_name ); ?>" type="hidden" />
								<input <?php checked( $field_value, 'yes' ); ?> value="yes" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" type="checkbox" />
								<span class="switch"></span>
							</label>
						</span>
						<p class="description"><label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html__( 'Show all your "', 'wp-travel-pro' ) . esc_html( $field_label ) . esc_html__( '" settings and enable its feature.', 'wp-travel-pro' ); ?></label></p>
					</td>
				</tr>
			</table>
			<?php

		}

		public static function plugin_disabled_notice() {
			echo sprintf( '<tr><td colspan="2"><p><strong>WP Travel Tour Extras</strong> is currently disabled please go to <a href="%s">Addons Settings</a> to enable it. </p><td></tr>', esc_url( admin_url( 'edit.php?post_type=itinerary-booking&page=settings#wp-travel-tab-content-addons_settings' ) ) );
		}
		/**
		 * Filter Post type args for Tour Extra.
		 */
		public static function register_tour_extras( $args, $post_type ) {

			// If not Tour Extras, bail.
			if ( 'tour-extras' === $post_type ) {
				$args['public']             = true;
				$args['publicly_queryable'] = true;
				// $args['supports']           = array( 'title', 'editor', 'thumbnail' );
			}
			return $args;
		}


		/**
		 * Utilities Frontend assets.
		 */
		public static function frontend_assets() {

			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			// wp_enqueue_style( 'wp-travel-tour-extras-slick-css', self::$plugin_url . 'assets/css/lib/slick/slick.min.css', '', self::$version );

			wp_register_style( 'wp-travel-tour-extras-front-end-css', self::$plugin_url . 'assets/css/wp-travel-tour-extras-front-end' . $suffix . '.css', array( 'wp-travel-slick' ) );

			// wp_enqueue_script( 'wp-travel-tour-extras-slick-js', self::$plugin_url . 'assets/js/lib/slick.min.js', array( 'jquery' ), '1.8.0', true );
			// $slick_handler = 'wp-travel-tour-extras-slick-js';
			wp_register_script( 'wp-travel-tour-extras-front-end-js', self::$plugin_url . 'assets/js/wp-travel-tour-extras-front-end' . $suffix . '.js', array( 'jquery', 'wp-travel-slick' ), self::$version, true );

			wp_register_script( 'wp-travel-tour-extras-cart-js', self::$plugin_url . 'assets/js/cart.js', array( 'jquery' ), self::$version, true );

			$cart_localize_strings = array(
				'confirm' => __( 'Are you sure to remove?', 'wp-travel-pro' ),
			);

			wp_localize_script( 'wp-travel-tour-extras-cart-js', 'extras_cart_texts', $cart_localize_strings );

			if ( method_exists( 'WP_Travel', 'is_page' ) ) { // @since WP Travel 4.4.2
				$is_cart_page = WP_Travel::is_page( 'cart' );
			} else {
				$is_cart_page = wp_travel_is_cart_page();
			}
			if ( $is_cart_page ) {
				wp_enqueue_script( 'wp-travel-tour-extras-cart-js' );
			}

			wp_enqueue_style( 'jquery-magnific-popup-css', self::$plugin_url . 'assets/css/magnific-popup.min.css' );

			wp_register_style( 'wp-travel-tour-extras-front-end-css', self::$plugin_url . 'assets/css/wp-travel-tour-extras-front-end' . $suffix . '.css', array( 'wp-travel-slick', 'wp-travel-popup' ) );

			if ( is_singular( 'tour-extras' ) ) {
				wp_enqueue_script( 'wp-travel-tour-extras-front-end-js' );
				wp_enqueue_style( 'wp-travel-tour-extras-front-end-css' );
				if ( ! wp_script_is( 'travel-door-popup' ) ) {
					wp_enqueue_script( 'wp-travel-popup' );
					wp_enqueue_script( 'jquery-magnific-popup-js', self::$plugin_url . 'assets/js/jquery.magnific-popup.min.js', array( 'jquery' ), self::$version, true );
				}
			}


		}

		/**
		 * Utilities admin assets.
		 */
		public static function admin_assets() {

			$suffix         = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			$screen         = get_current_screen();
			$allowed_screen = array( 'edit-itineraries', WP_TRAVEL_POST_TYPE, 'itineraries_page_settings', 'itinerary-booking', 'itinerary-booking_page_settings' );

			// wp_enqueue_script( 'wp-travel-tour-extras-back-end-js', self::$plugin_url . 'assets/js/wp-travel-tour-extras-back-end' . $suffix . '.js', array( 'jquery' ), self::$version, true );

			// wp_enqueue_style( 'wp-travel-tour-extras-back-end-css', self::$plugin_url . 'assets/css/wp-travel-tour-extras-back-end' . $suffix . '.css' );

			$allowed_screen = array( 'tour-extras', 'edit-tour-extras' );

			$screen = get_current_screen();

			if ( in_array( $screen->id, $allowed_screen ) ) {

				wp_enqueue_script( 'wp-travel-tour-extras-jquery-media-upload', self::$plugin_url . 'assets/js/jquery.wptraveluploader' . $suffix . '.js', array( 'jquery' ), self::$version, 1 );

				wp_register_script( 'wp-travel-tour-extras-media-upload', self::$plugin_url . 'assets/js/wp-travel-tour-extras-media-upload' . $suffix . '.js', array( 'jquery', 'plupload-handlers', 'jquery-ui-sortable', 'jquery-datepicker-lib', 'jquery-datepicker-lib-eng' ), self::$version, 1 );

				// $wp_travel_tour_extras_gallery_data = array(
				// 	'ajax'            => admin_url( 'admin-ajax.php' ),
				// 	'labels'          => array(
				// 		'uploader_files_computer' => __( 'Select Files from Your Computer', 'wp-travel-pro' ),
				// 	),
				// 	'drag_drop_nonce' => wp_create_nonce( 'wp-travel-drag-drop-nonce' ),
				// );

				// $wp_travel_tour_extras_gallery_data = apply_filters( 'wp_travel_tour_extras_localize_gallery_data', $wp_travel_tour_extras_gallery_data );
				// wp_localize_script( 'wp-travel-tour-extras-media-upload', 'wp_travel_drag_drop_uploader', $wp_travel_tour_extras_gallery_data );

				$wp_travel_tour_extras_gallery_data = array(
					'ajax'            => admin_url( 'admin-ajax.php' ),
					'labels'          => array(
						'uploader_files_computer' => __( 'Select Files from Your Computer', 'wp-travel-pro' ),
					),
					'drag_drop_nonce' => wp_create_nonce( 'wp-travel-drag-drop-nonce' ),
				);
				
				$wp_travel_tour_extras_gallery_data = apply_filters( 'wp_travel_tour_extras_localize_gallery_data', $wp_travel_tour_extras_gallery_data );
				
				// JSON encode the data
				$script_data = 'var wp_travel_drag_drop_uploader = ' . json_encode( $wp_travel_tour_extras_gallery_data ) . ';';
				
				wp_add_inline_script( 'wp-travel-tour-extras-media-upload', $script_data, 'before' );

				wp_enqueue_script( 'wp-travel-tour-extras-media-upload' );
			}
			// React Addons Settings Page.
			if ( WP_Travel::is_page( 'settings', true ) ) {
				$deps = include_once sprintf( '%sapp/build/settings.asset.php', self::$abspath );
				wp_enqueue_script( 'wptravel-extras-settings', self::$plugin_url . 'app/build/settings.js', $deps['dependencies'], $deps['version'], true );
			}
		}

		public static function trip_extra_fields( $post_id, $context, $fetch_key, $table_row ) {

			$name = 'wp_travel_tour_extras_partial';
			if ( $context && 'pricing_options' === $context ) {
				$pricing_options     = get_post_meta( $post_id, 'wp_travel_pricing_options', true );
				$trip_extras_partial = isset( $pricing_options[ $fetch_key ]['tour_extras_partial'] ) && ! empty( $pricing_options[ $fetch_key ]['tour_extras_partial'] ) ? $pricing_options[ $fetch_key ]['tour_extras_partial'] : false;
				$name                = 'wp_travel_pricing_options[' . $fetch_key . '][tour_extras_partial]';
			} elseif ( ! $context && 'wp_travel_tour_extras' === $fetch_key ) {
				$trip_extras_partial = get_post_meta( $post_id, 'wp_travel_tour_extras_partial', true );
			}

			if ( $table_row ) :
				?>
				</tr>
				<tr class="price-option-row single-price-option-row wp-travel-tour-extra-content">
				<td>
				<?php
			else :
				?>
				</div>
				<div class="repeat-row">
				<div><div class="one-third">
				<?php
			endif;
			?>
				<label for=""><?php echo esc_html( 'Enable partial for extras' ); ?></label>
			<?php
			if ( $table_row ) :
				?>
				</td><td>
				<?php
			else :
				?>
				</div><div class="two-third">
				<?php
			endif;
			?>

				<span class="show-in-frontend checkbox-default-design">
					<label data-on="ON" data-off="OFF">
						<input name="<?php echo esc_attr( $name ); ?>" type="checkbox" id="<?php echo esc_attr( $name . '-' . $fetch_key ); ?>" <?php checked( $trip_extras_partial, 1 ); ?> value="1" />
						<span class="switch"></span>
					</label>
				</span>
				<p class="wp-travel-enable-sale description"><?php esc_html_e( 'Check to enable partial payment for trip extras.', 'wp-travel-pro' ); ?></p>
				<?php
				if ( $table_row ) :
					?>
				</td>
					<?php
			else :
				?>
				</div></div>
				<?php
			endif;
		}

		public static function save_meta_data( $trip_id ) {
			if ( ! $trip_id ) {
				return;
			}
			$meta = '';
			if ( isset( $_POST['wp_travel_tour_extras_partial'] ) ) {
				$meta = sanitize_text_field( wp_unslash( $_POST['wp_travel_tour_extras_partial'] ) );
			}
			update_post_meta( $trip_id, 'wp_travel_tour_extras_partial', $meta );
		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}
			/**
			 * Include required files
			 */
			include sprintf( '%sinc/functions.php', self::$abspath );
			include sprintf( '%sinc/admin/class-tour-extras-post-type-hooks.php', self::$abspath );
			include sprintf( '%sinc/template-functions.php', self::$abspath );
			include sprintf( '%sinc/public/class-extras-template.php', self::$abspath );
			include sprintf( '%sinc/class-ajax.php', self::$abspath );

			if ( self::is_request( 'frontend' ) ) {
				include sprintf( '%sinc/public/class-tour-extras-frontend.php', self::$abspath );
			}
		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}


	}
endif;
