<?php
/**
 * WP Travel Stripe Checkout Core Class.
 *
 * @package wp-travel-stripe-checkout-core
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Stripe Checkout core.
if ( ! class_exists( 'WP_Travel_Stripe_Checkout_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Stripe_Checkout_Core {

		const WP_TRAVEL_STRIPE_HANDLE = 'wp_travel_stripe_';
		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Stripe Checkout Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Stripe_Checkout_Core Instance.
		 * Ensures only one instance of WP_Travel_Stripe_Checkout_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Stripe_Checkout_Core()
		 * @return WP_Travel_Stripe_Checkout_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];

			self::includes();

			add_action( 'wp_enqueue_scripts', array( 'WP_Travel_Stripe_Checkout_Core', 'frontend_assets' ), 20 );
			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Stripe_Checkout_Core', 'admin_assets' ), 20 );

			add_action( 'wp_ajax_nopriv_stripe_charge', 'wp_travel_stripe_charge', 20 );
			add_action( 'wp_ajax_stripe_charge', 'wp_travel_stripe_charge', 20 );

			add_action( 'wp_ajax_nopriv_payment_intents', 'wp_travel_stripe_payment_intent', 20 );
			add_action( 'wp_ajax_payment_intents', 'wp_travel_stripe_payment_intent', 20 );

			// Payment Gateway list.
			add_filter( 'wp_travel_payment_gateway_lists', 'wp_travel_gateway_stripe' );
			add_filter( 'wp_travel_premium_addons_list', 'wp_travel_stripe_addons' );

			if ( self::is_enabled() ) {
				add_filter( 'wptravel_frontend_data', 'wp_travel_stripe_add_vars', 10, 2 );
			}

			add_action( 'wp_travel_action_after_payment_info_field', array( __CLASS__, 'add_extra_fields' ) );
			add_action( 'wp_travel_dashboard_booking_after_detail', array( __CLASS__, 'add_extra_fields' ), 15, 2 );

			// add_action( 'wp_travel_action_after_payment_info_field', array( __CLASS__, 'add_extra_fields' ) );
			add_filter( 'wptravel_frontend_data', array( __CLASS__, 'wp_travel_localize_frontend_string') );
			if ( self::uses_stripe_checkout() ) {
				// This will call after payement completed.
				add_action( 'wp_travel_after_frontend_booking_save', array( 'WP_Travel_Stripe_Checkout_Core', 'set_up_payment_options' ), 20 );
				add_action( 'wp_travel_after_partial_payment_complete', array( 'WP_Travel_Stripe_Checkout_Core', 'set_up_partial_payment' ), 20 );
			}

			if( isset( wptravel_get_settings()['stripe_new_payment_ui'] ) && wptravel_get_settings()['stripe_new_payment_ui'] == 'yes' ){ 
				add_action( 'wp_travel_after_partial_payment_complete', array( 'WP_Travel_Stripe_Checkout_Core', 'set_up_partial_payment' ), 20 );
			}
			
			isset( $_SESSION['used-stripe'] ) && $_SESSION['used-stripe'] && add_filter( 'wp_travel_booked_message', 'wp_travel_stripe_booking_message', 20 );

		}
		/**
		 * @since 5.7.0
		 * Localize card string
		 */
		public static function wp_travel_localize_frontend_string( $wp_travel ) {
			$wp_travel['strip_card'] = __( 'Card', 'wp-travel-pro' );
			return $wp_travel;
		}

		public static function add_extra_fields( $booking_id = null, $details = null ) {
			?>
			<div id="card-errors"></div>
			<?php
		}

		/**
		 * Determine if booking used express checkout.
		 */
		private static function uses_stripe_checkout() {
			$settings       = wptravel_get_settings();
			if( isset( $settings['selected_booking_option'] ) && count( $settings['selected_booking_option'] ) == 1 && $settings['selected_booking_option'][0] = 'booking-with-payment' ){
			return 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'stripe' === $_REQUEST['wp_travel_payment_gateway'];
			}else{
				return isset( $_POST['wp_travel_booking_option'] ) && 'booking_with_payment' === $_POST['wp_travel_booking_option'] && 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'stripe' === $_REQUEST['wp_travel_payment_gateway'];
			}
		
		}

		/**
		 * Determine if Stripe checkout is enabled.
		 */
		private static function is_enabled() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			return array_key_exists( 'payment_option_stripe', $settings ) && 'yes' === $settings['payment_option_stripe'];
		}

		/**
		 * Determing is Stripe checkout is disabled.
		 */
		private static function is_disabled() {
			return ! self::is_enabled();
		}

		/**
		 * Stripe Frontend assets.
		 */
		public static function frontend_assets() {

			if ( function_exists( 'wptravel_can_load_payment_scripts' ) ) {
				$load_payment_script = wptravel_can_load_payment_scripts();
			} else {
				$load_payment_script = wp_travel_can_load_payment_scripts();
			}
			if ( $load_payment_script && self::is_enabled() ) {

				$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
				// Styles.
				wp_enqueue_style( 'wp-travel-stripe-custom', self::$plugin_url . 'assets/css/custom.css', '', self::$version );

				$dependencies   = array( 'jquery', 'wp-travel-payment-frontend-script' );
				$dependencies[] = 'wp-travel-payment-frontend-script';
				// wp_enqueue_script( 'wp_travel_stripe_checkout_js', 'https://checkout.stripe.com/checkout.js', array( 'jquery' ), self::$version, true );
				wp_enqueue_script( 'wp_travel_stripe_checkout_js', 'https://js.stripe.com/v3/', array( 'jquery' ), self::$version, true );

				if( isset( wptravel_get_settings()['stripe_new_payment_ui'] ) && wptravel_get_settings()['stripe_new_payment_ui'] == 'yes' ){
					// wp_register_script( 'wp-travel-formsaver-script', self::$plugin_url . 'assets/js/formsaver.js', $dependencies, self::$version, true );
					wp_register_script( 'wp-travel-stripe-script', self::$plugin_url . 'assets/js/wp-travel-new-stripe.js', $dependencies, self::$version, true );
				}else{
					wp_register_script( 'wp-travel-stripe-script', self::$plugin_url . 'assets/js/wp-travel-stripe.js', $dependencies, self::$version, true );
				}
				
				wp_localize_script(
					'wp-travel-stripe-script',
					'_wpTravelStripeL10n',
					array(
						'brandLogo' => plugin_dir_url( __FILE__ ) . 'assets/img/powered_by_stripe@2x.png',
						'_nonce'                        => wp_create_nonce( 'wp_travel_nonce' ),
						'pay_with_stripe' => __( 'Pay with stripe', 'wp-travel-pro' ),
						'proceed_payment' => __( 'Proceed Payment', 'wp-travel-pro' )
					)
				);
				// wp_enqueue_script( 'wp-travel-formsaver-script' );
				wp_enqueue_script( 'wp-travel-stripe-script' );
			}
			global $post;
			// wp-travel-authorize-net-checkout-frontend
			if ( get_post_type() == 'itineraries' ) {
				wp_register_script( 'wp_travel_stripe_booking_js', 'https://js.stripe.com/v3/', array( 'jquery' ), self::$version, true );
				$depend = array( 'wp-travel-frontend-booking-widget', 'wp_travel_stripe_booking_js' );
				wp_register_script( 'wp-travel-stripe-parsley-script', self::$plugin_url . 'assets/js/parsley.js', $depend, self::$version, true );
				wp_register_script( 'wp-travel-stripe-frontend-script', self::$plugin_url . 'app/build/wp-travel-frontend-booking.js', $depend, self::$version, true );
				wp_localize_script(
					'wp-travel-stripe-frontend-script',
					'_wpTravelStripeL10n',
					array(
						'brandLogo' => plugin_dir_url( __FILE__ ) . 'assets/img/powered_by_stripe@2x.png',
						'_nonce'                        => wp_create_nonce( 'wp_travel_nonce' ),
					)
				);
				wp_enqueue_script( 'wp-travel-stripe-parsley-script' );
				wp_enqueue_script( 'wp-travel-stripe-frontend-script' );
			}
		}

		/**
		 * Admin assets.
		 */
		public static function admin_assets() {

			$suffix         = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			$screen         = get_current_screen();
			$allowed_screen = array( 'edit-itineraries', WP_TRAVEL_POST_TYPE, 'itineraries_page_settings', 'itinerary-booking', 'itinerary-booking_page_settings' );

			$screen = get_current_screen();
			if (  method_exists( 'WP_Travel',  'is_page' )  ) { // @since WP Travel 4.4.2
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_STRIPE_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
			}
		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}
			/**
			 * Include required files
			 */
			if ( ! class_exists( 'Stripe\Stripe' ) ) {
				if( isset( wptravel_get_settings()['stripe_new_payment_ui'] ) && wptravel_get_settings()['stripe_new_payment_ui'] == 'yes' ){
					
					require_once sprintf( '%s/new-vendor/autoload.php', self::$abspath );
				}else{
					require_once sprintf( '%s/vendor/stripe/init.php', self::$abspath );
				}
				
			}

			include sprintf( '%s/inc/functions.php', self::$abspath );

			$stripe_key = apply_filters( 'new_stripe_key', false );

			if( isset( wptravel_get_settings()['stripe_new_payment_ui'] ) && wptravel_get_settings()['stripe_new_payment_ui'] == 'yes' ){
				
				include sprintf( '%s/inc/wp-travel-stripe-new-function.php', self::$abspath );
			}else{
				if( $stripe_key ){
					include sprintf( '%s/inc/wp-travel-stripe-function-v2.php', self::$abspath );
				}else{
					include sprintf( '%s/inc/wp-travel-stripe-function.php', self::$abspath );
				}
			}
	
			if ( self::is_request( 'admin' ) ) {
				include sprintf( '%s/inc/admin/settings.php', self::$abspath );
			}
		}

		/**
		 * Sets up payment options
		 *
		 * @param string $booking_id ID of booking.
		 * @return void
		 */
		public static function set_up_payment_options( $booking_id ) {

			if ( self::is_disabled() ) {
				return;
			}

			if ( ! self::uses_stripe_checkout() ) {
				return;
			}

			if ( ! $booking_id ) {
				return;
			}
			if ( ! isset( $_POST['wp_travel_book_now'] ) ) {
				return;
			}

			do_action( 'wt_before_payment_process', $booking_id );

			$json = sanitize_text_field( wp_unslash( $_REQUEST['payment_details'] ) );

			$detail   = json_decode( $json );

			if( isset( wptravel_get_settings()['stripe_new_payment_ui'] ) && wptravel_get_settings()['stripe_new_payment_ui'] == 'yes' ){
				$amount   = isset( $detail->amount ) ? $detail->amount : 0; // @since 1.0.7
				$currency = isset( $detail->currency ) ? strtoupper( $detail->currency ) : 'USD'; // @since 1.0.7
			}else{
				$amount   = isset( $detail->paymentIntent->amount ) ? $detail->paymentIntent->amount : 0; // @since 1.0.7
				$currency = isset( $detail->paymentIntent->currency ) ? strtoupper( $detail->paymentIntent->currency ) : 'USD'; // @since 1.0.7
			}

			if ( $amount ) {
				$amount = wp_travel_stripe_revert_amount( $amount, $currency );
			}

			$payment_id     = get_post_meta( $booking_id, 'wp_travel_payment_id', true );
			$payment_gateway = 'stripe';
			update_post_meta( $payment_id, 'wp_travel_payment_gateway', $payment_gateway );

			wptravel_update_payment_status( $booking_id, $amount, 'paid', $detail, sprintf( '_%s_args', $payment_gateway ), $payment_id );
			$_SESSION['used-stripe'] = true;
			do_action( 'wp_travel_after_successful_payment', $booking_id );
		}

		/**
		 * Sets up payment options
		 *
		 * @return void
		 */
		public static function set_up_partial_payment() {
			
			if ( self::is_disabled() ) {
				return;
			}

			if( isset( $_REQUEST['payment_intent_client_secret'] ) ){
				$payment_gateway = 'stripe';
				$booking_id      = sanitize_text_field( wp_unslash( $_POST['wp_travel_booking_id'] ) );
				// get previous payment ids.
				$payment_id = get_post_meta( $booking_id, 'wp_travel_payment_id', true );
				/**
				 * This will return the current payment id to insert payment meta for the current payment.
				 * 
				 * @hooked wptravel_update_payment_post from partial payment which will also update booking option [booking_only | booking_with_payment ]
				 *
				 * @since 4.0.2
				 */
				
				$new_payment_id = apply_filters( 'wptravel_before_insert_partial_payment', $payment_id, $booking_id, $payment_gateway );
				
				
				$json = sanitize_text_field( wp_unslash( $_POST['payment_details'] ) );

				$detail = json_decode( $json );
				$amount = isset( $detail->amount ) ? $detail->amount : 0; // @since 1.0.7

				if ( $amount ) {
					$amount = number_format( $amount / 100, 2, '.', '' );
				}

				wptravel_update_payment_status( $booking_id, $amount, 'paid', $detail, sprintf( '_%s_args', $payment_gateway ), $new_payment_id );
				$_SESSION['used-stripe'] = true;
			}else{
				if ( ! self::uses_stripe_checkout() ) {
					return;
				}
				
				if ( ! isset( $_POST['complete_partial_payment'] ) ) {
					return;
				}
				
				if ( ! isset( $_POST['wp_travel_booking_id'] ) ) {
					return;
				}

				$payment_gateway = sanitize_text_field( wp_unslash( $_POST['wp_travel_payment_gateway'] ) );
				$booking_id      = sanitize_text_field( wp_unslash( $_POST['wp_travel_booking_id'] ) );
				// get previous payment ids.
				$payment_id = get_post_meta( $booking_id, 'wp_travel_payment_id', true );
				/**
				 * This will return the current payment id to insert payment meta for the current payment.
				 * 
				 * @hooked wptravel_update_payment_post from partial payment which will also update booking option [booking_only | booking_with_payment ]
				 *
				 * @since 4.0.2
				 */
				
				$new_payment_id = apply_filters( 'wptravel_before_insert_partial_payment', $payment_id, $booking_id, $payment_gateway );
				
				
				$json = sanitize_text_field( wp_unslash( $_POST['payment_details'] ) );

				$detail = json_decode( $json );
				$amount = isset( $detail->paymentIntent->amount ) ? $detail->paymentIntent->amount : 0; // @since 1.0.7

				if ( $amount ) {
					$amount = number_format( $amount / 100, 2, '.', '' );
				}
				wptravel_update_payment_status( $booking_id, $amount, 'paid', $detail, sprintf( '_%s_args', $payment_gateway ), $new_payment_id );
				$_SESSION['used-stripe'] = true;
			}
		}
		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}
	}
endif;
