<?php
/**
 * Stripe Functions.
 *
 * @package wp-travel-stripe-ideal/
 */

/**
 * Booking Message.
 *
 * @param string $message Message String.
 * @return string Booking Message.
 */
function wp_travel_stripe_ideal_booking_message( $message ) {
	unset( $_SESSION['used-stripe'] );
	$message = esc_html__( "We've received your booking and payment details. We'll contact you soon.", 'wp-travel-pro' );
	return $message;
}

add_action( 'init', 'wp_travel_stripe_ideal_webhook_handler' );
/**
 * Stripe iDEAL Webhook Handler.
 *
 * @return void
 */
function wp_travel_stripe_ideal_webhook_handler() {

	if ( ! isset( $_GET['webhook'] ) || 'ideal-source' !== wp_unslash( $_GET['webhook'] ) ) {
		return;
	}

	$payload = @file_get_contents( 'php://input' );
	$event   = null;

	try {
		$event = \Stripe\Event::constructFrom(
			json_decode( $payload, true )
		);
	} catch ( \UnexpectedValueException $e ) {
		// Invalid payload.
		http_response_code( 400 );
		exit();
	}

	$source_id  = 'charge.succeeded' === $event->type ? $event->data->object->source->id : $event->data->object->id;
	$booking_id = get_transient( 'bid_' . $source_id );
	$payment_id = get_transient( 'pid_' . $source_id );
	$args       = is_array( get_post_meta( $payment_id, '_stripe_ideal_args', true ) ) ? get_post_meta( $payment_id, '_stripe_ideal_args', true ) : [];
	$response   = [];
	
	// Handle the event.
	switch ( $event->type ) {
		case 'source.chargeable':
			$stripe_ideal_args = $args;
			if ( is_array( $stripe_ideal_args ) ) {
				$stripe_ideal_args['status'] = $event->data->object->status;

				$response['event_id']       = $event->id;
				$response['source_id']      = $event->data->object->id;
				$response['amount']         = $event->data->object->amount;
				$response['source_created'] = $event->data->object->created;

				$stripe_ideal_args['source.chargeable'] = $response;
			}
			$secret_key = wp_travel_stripe_ideal_get_api_secret_key();
			\Stripe\Stripe::setApiKey( $secret_key );
			try {
				\Stripe\Charge::create(
					[
						'amount'   => $event->data->object->amount,
						'currency' => 'eur',
						'source'   => $event->data->object->id,
					]
				);
			} catch ( \Exception $e ) {
				$stripe_ideal_args['status'] = __( 'Charge request failed : ', 'wp-travel-pro' ) . $e->getMessage();
			}
			update_post_meta( $payment_id, '_stripe_ideal_args', $stripe_ideal_args, $args );
			break;
		case 'source.canceled':
			if ( is_array( $args ) ) {
				$args['status']          = 'canceled';
				$args['source.canceled'] = [
					'event_id'  => $event->id,
					'source_id' => $event->data->object->id,
				];
				update_post_meta( $payment_id, '_stripe_ideal_args', $args );
				update_post_meta( $payment_id, 'wp_travel_payment_status', 'canceled' );
			}
			break;
		case 'source.failed':
			if ( is_array( $args ) ) {
				$args['status']        = 'failed';
				$args['source.failed'] = [
					'event_id'  => $event->id,
					'source_id' => $event->data->object->id,
				];
				update_post_meta( $payment_id, '_stripe_ideal_args', $args );
				update_post_meta( $payment_id, 'wp_travel_payment_status', 'canceled' );
			}
			break;
		case 'charge.pending':
			$args['status']        = 'charge.pending';
			$response['event_id']  = $event->id;
			$response['charge_id'] = $event->data->object->id;
			$response['amount']    = $event->data->object->amount;

			$args['charge.pending'] = $response;
			update_post_meta( $payment_id, '_stripe_ideal_args', $args );
			update_post_meta( $payment_id, 'wp_travel_payment_status', 'pending' );
			break;
		case 'charge.failed':
			$args['status']        = 'charge.failed';
			$response['event_id']  = $event->id;
			$response['charge_id'] = $event->data->object->id;
			$response['amount']    = $event->data->object->amount;

			$args['charge.failed'] = $response;
			update_post_meta( $payment_id, '_stripe_ideal_args', $args );
			update_post_meta( $payment_id, 'wp_travel_payment_status', 'pending' );
			break;
		case 'charge.succeeded':
			$args['status']        = 'charge.succeeded';
			$response['event_id']  = $event->id;
			$response['charge_id'] = $event->data->object->id;
			$response['source_id'] = $event->data->object->source->id;
			$response['amount']    = $event->data->object->amount;

			$args['charge.succeeded'] = $response;
			update_post_meta( $payment_id, '_stripe_ideal_args', $args );
			update_post_meta( $payment_id, 'wp_travel_payment_status', 'paid' );
			delete_transient( 'bid_' . $event->data->object->source->id );
			delete_transient( 'pid_' . $event->data->object->source->id );
			do_action( 'wp_travel_after_successful_payment', $booking_id );
			break;
		default:
			http_response_code( 400 );
			exit();
	}
	http_response_code( 200 );
	wp_send_json_success( [ 'payment_id' => $payment_id, 'booking_id' => $booking_id, 'args' => $args ] );
	exit;
}

/**
 * Gets API Secret Key.
 *
 * @return void
 */
function wp_travel_stripe_ideal_get_api_secret_key() {
	if ( function_exists( 'wptravel_get_settings' ) ) {
		$settings = wptravel_get_settings();
	} else {
		$settings = wp_travel_get_settings();
	}
	if ( function_exists( 'wptravel_test_mode' ) ) {
		$test_mode = wptravel_test_mode();
	} else {
		$test_mode = wp_travel_test_mode();
	}

	if ( $test_mode ) {
		return ! empty( $settings['stripe_ideal_test_secret_key'] ) ? $settings['stripe_ideal_test_secret_key'] : null;
	} else {
		return ! empty( $settings['stripe_ideal_secret_key'] ) ? $settings['stripe_ideal_secret_key'] : null;
	}
}

/**
 * Default Webhook URL.
 *
 * @return void
 */
function wp_travel_stripe_ideal_default_webhook_url() {
	$url = esc_url(
		add_query_arg(
			array(
				'webhook' => 'ideal-source',
			),
			home_url( 'index.php' )
		)
	);
	return apply_filters( 'stripe_ideal_webhook_url', $url );
}

/**
 * Retrieves the webhook url if exists and return the url.
 *
 * @return mixed.
 */
function wp_travel_stripe_ideal_get_webhook_url() {
	$secret_key = wp_travel_stripe_ideal_get_api_secret_key();
	
	if ( is_null( $secret_key ) ) {

		if ( function_exists( 'wptravel_test_mode' ) ) {
			$test_mode = wptravel_test_mode();
		} else {
			$test_mode = wp_travel_test_mode();
		}

		$message      = '';
		$message      = $test_mode ? __( 'Please update settings with Test API Keys to register webhook.', 'wp-travel-pro' ) : __( 'Please update settings with LIVE API Keys to register webhook. ', 'wp-travel-pro' );
		$api_keys_url = $test_mode ? 'https://dashboard.stripe.com/test/apikeys' : 'https://dashboard.stripe.com/apikeys';
		$message     .= __( ' Get Your API Keys ', 'wp-travel-pro' ) . '<a href="' . $api_keys_url . '">' . __( 'Here', 'wp-travel-pro' ) . '</a>';
		if ( wp_doing_ajax() ) {
			wp_send_json_error( array( 'secret_key_na' => $message ) );
		}
		return array( 'secret_key_na' => $message );
	}

	if ( ! empty( get_option( 'stripe_ideal_webhook', '' ) ) ) {
		$results = [
			'url'         => get_option( 'stripe_ideal_webhook' )['url'],
			'status'      => get_option( 'stripe_ideal_webhook' )['status'],
			'statusclass' => get_option( 'stripe_ideal_webhook' )['status'],
		];

		if ( wp_doing_ajax() ) {
			wp_send_json_success( $results );
		}

		return $results;
	}
	if ( function_exists( 'wptravel_test_mode' ) ) {
		$test_mode = wptravel_test_mode();
	} else {
		$test_mode = wp_travel_test_mode();
	}
	$webhook_url = $test_mode ? 'https://dashboard.stripe.com/test/webhooks' : 'https://dashboard.stripe.com/webhooks';

	\Stripe\Stripe::setApiKey( $secret_key );

	try {
		$endpoint   = \Stripe\WebhookEndpoint::create(
			[
				'url'            => wp_travel_stripe_ideal_default_webhook_url(),
				'enabled_events' => [ 'charge.succeeded', 'source.chargeable', 'source.failed', 'source.canceled' ],
			]
		);
		$webhook_id = isset( $endpoint->getLastResponse()->json['id'] ) ? $endpoint->getLastResponse()->json['id'] : '';

		update_option(
			'stripe_ideal_webhook',
			[
				'id'     => $webhook_id,
				'status' => $endpoint->getLastResponse()->json['status'],
				'url'    => $endpoint->getLastResponse()->json['url'],
			]
		);

		$results = array();
		$results = [
			'url'         => $endpoint->getLastResponse()->json['url'],
			'status'      => $endpoint->getLastResponse()->json['status'],
			'message'     => __( 'Webhook Added successfully. View your webhook endpoints List ', 'wp-travel-pro' ) . '<a target="new" href="' . $webhook_url . '">' .__( 'here', 'wp-travel-pro' ). '</a>' . __( ' and confirm that ', 'wp-travel-pro' ) . '<code>' .wp_travel_stripe_ideal_default_webhook_url(). '</code>' . __( ' registered and active.', 'wp-travel-pro' ),
			'class'       => 'success',
			'statusclass' => $endpoint->getLastResponse()->json['status'],
		];

		if ( wp_doing_ajax() ) {
			wp_send_json_success( $results );
		}
		return $results;

	} catch ( Exception $e ) {

		$results = array(
			'url'     => wp_travel_stripe_ideal_default_webhook_url(),
			'message' => $e->getMessage() . ' <br>Webhook Endpoint couldn\'t be added. Please verify the API Credentials and try again.',
			'class'   => 'warning',
		);

		if ( wp_doing_ajax() ) {
			wp_send_json_error( $results );
		}

		return $results;
	}
}
/**
 * Added for v4 settings support.
 */
add_action( 'wp_ajax_wp_travel_stripe_ideal_get_webhook_url', 'wp_travel_stripe_ideal_get_webhook_url' );

# redirect to thankyou page with booking id
add_action( 'init', 'wp_travel_stripe_ideal_redirect_with_booking_id', 999999 );
function wp_travel_stripe_ideal_redirect_with_booking_id(){

	if( isset( $_GET[ 'booked' ], $_GET[ 'source' ] ) ){

		$source_id  = $_GET[ 'source' ];
		$booking_id = get_transient( 'bid_' . $source_id );

		if( $booking_id > 0 ){

			$thankyou_page_url = wptravel_thankyou_page_url();
			$thankyou_page_url = add_query_arg( 'booked', true, $thankyou_page_url );
			$thankyou_page_url = add_query_arg( 'order_id', $booking_id, $thankyou_page_url );

			wp_redirect( $thankyou_page_url );
			exit;
		}
	}
}