<?php
/**
 * Modules core file.
 *
 * @package wp-travel-payu-checkout-core
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

// WP Travel PayU Checkout Core.
if ( ! class_exists( 'WP_Travel_PayU_Checkout_Core' ) ) :
	/**
	 * WP Travel PayU Checkout Core.
	 */
	class WP_Travel_PayU_Checkout_Core {

		const WP_TRAVEL_PAYU_CHECKOUT_HANDLE = 'wp_travel_payu_checkout_';

		/**
		 * Absolute path to core
		 *
		 * @var string
		 */
		protected static $abspath;

		/**
		 * Plugin File.
		 *
		 * @var [type]
		 */
		protected static $plugin_file;

		/**
		 * Plugin Version.
		 *
		 * @var string
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP_Travel_PayU_Checkout_Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_PayU_Checkout_Core Instance.
		 * Ensures only one instance of WP_Travel_PayU_Checkout_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_PayU_Checkout_Core()
		 * @return WP_Travel_PayU_Checkout_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param Array $params Core class init paramerters.
		 */
		public static function init( $params ) {
			self::$abspath     = $params['abspath'] . 'inc/modules/payu-checkout-core/';
			self::$plugin_file = __FILE__;
			self::$version     = $params['version'];
			self::includes();
			self::init_hooks();
		}

		/**
		 * Includes required files.
		 */
		public static function includes() {
			include_once self::$abspath . 'inc/lib/openpayu.php';
			include_once self::$abspath . 'inc/admin/settings.php';
			include_once self::$abspath . 'inc/payu/class-wp-travel-payu-payment.php';
			include_once self::$abspath . 'inc/functions.php';
		}

		/**
		 * Init Hooks.
		 */
		public static function init_hooks() {
			add_action( 'wp_enqueue_scripts', array( __CLASS__, 'assets' ) );
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'admin_assets' ) );
			if ( self::uses_payu_checkout() ) {
				add_action( 'wp_travel_after_frontend_booking_save', array( __CLASS__, 'process' ), 20, 1 );
				add_action( 'wp_travel_before_partial_payment_complete', array( __CLASS__, 'process' ), 10, 2 );
			}

			// General Notices.
			add_filter( 'wp_travel_display_general_admin_notices', array( __CLASS__, 'wp_travel_display_payu_notices' ), 20 );
			add_action( 'wp_travel_general_admin_notice', array( __CLASS__, 'wp_travel_payu_notices' ), 20 );

			if ( isset( $_SESSION['used-payu'] ) && $_SESSION['used-payu'] ) {
				add_filter( 'wp_travel_booked_message', array( __CLASS__, 'booking_message' ), 20 );
			}

		}

		/**
		 * Process After Checkout Form Submit.
		 *
		 * @param int     $booking_id Booking ID.
		 * @param boolean $complete_partial_payment Complete Payment Process.
		 * @return void
		 */
		public static function process( $booking_id, $complete_partial_payment = false ) {
			if ( ! $booking_id || self::is_disabled() || ! self::uses_payu_checkout() ) {
				return;
			}

			if ( ! $complete_partial_payment ) {
				do_action( 'wt_before_payment_process', $booking_id );
			}

			wp_travel_payu_config();
			$order = self::get_order_data( $booking_id, $complete_partial_payment );

			try {
				$response    = OpenPayU_Order::create( $order );
				$status_desc = OpenPayU_Util::statusDesc( $response->getStatus() );
				if ( $response->getStatus() == 'SUCCESS' ) {
					wp_redirect( $response->getResponse()->redirectUri );
					exit;
				}
			} catch ( OpenPayU_Exception $e ) {
				error_log( (string) $e );
			}

		}

		/**
		 * Prepares order Data.
		 *
		 * @return array
		 */
		public static function get_order_data( $booking_id, $complete_partial_Payment ) {

			$payu_payment = new WP_Travel_PayU_Payment( $booking_id, $complete_partial_Payment );

			$order                = array();
			$order['notifyUrl']   = esc_url( add_query_arg( 'payu_listener', 'IPN', home_url( 'index.php' ) ) );
			$order['continueUrl'] = add_query_arg(
				array(
					'booking_id' => $booking_id,
					'booked'     => true,
					'status'     => 'success',
					'order_id'   => $booking_id,
				),
				$payu_payment->get_return_url()
			);

			$trans = $payu_payment->get_transaction_details();

			$order['customerIp']    = wp_travel_payu_get_user_ip();
			$order['merchantPosId'] = OpenPayU_Configuration::getOauthClientId() ? OpenPayU_Configuration::getOauthClientId() : OpenPayU_Configuration::getMerchantPosId();
			$order['description']   = "New Booking #{$booking_id} Payment";
			$order['currencyCode']  = $trans['currency'];
			$order['totalAmount']   = self::to_amount( $trans['amount'] );
			$order['extOrderId']    = uniqid( "{$booking_id}_", true );

			$order['products'][0]['name']      = $trans['items'];
			$order['products'][0]['unitPrice'] = self::to_amount( $trans['amount'] );
			$order['products'][0]['quantity']  = 1;

			$buyer = $payu_payment->get_buyer_details();

			$order['buyer']['email']     = $buyer['email'];
			$order['buyer']['phone']     = $buyer['phone'];
			$order['buyer']['firstName'] = $buyer['first_name'];
			$order['buyer']['lastName']  = $buyer['last_name'];
			$order['buyer']['language']  = substr_replace(  get_bloginfo( 'language' ), '', -3 );

			return $order;
		}

		/**
		 * @param $value
		 * @return int
		 */
		private static function to_amount( $value ) {
			return (int) round( $value * 100 );
		}

		/**
		 * Display admin notices in case of API credentials not found.
		 *
		 * @param $display
		 */
		public static function wp_travel_display_payu_notices( $display ) {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$currencies = array( 'PLN' );
			if ( in_array( $settings['currency'], $currencies ) && self::is_enabled() && ! $display ) {
				$display = true;
			}
			return $display;
		}

		/**
		 * Adds Notice if currency not matched.
		 */
		public static function wp_travel_payu_notices() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$currencies = array( 'PLN' );
			if ( in_array( $settings['currency'], $currencies ) && self::is_enabled() ) {
				$message = sprintf( __( 'PayU works only with Polish złoty(PLN).', 'wp-travel-pro' ) );
				printf( '<li ><p>%1$s</p></li>', esc_html( $message ) );
			}
		}

		/**
		 * Chekcs if this payment is disabled.
		 *
		 * @return boolean
		 */
		public static function is_disabled() {
			return ! self::is_enabled();
		}

		/**
		 * Checks if this payent is enabled.
		 *
		 * @return boolean
		 */
		private static function is_enabled() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			return isset( $settings['wp_travel_payu_settings'] ) && array_key_exists( 'payment_option_payu', $settings['wp_travel_payu_settings'] ) && 'yes' === $settings['wp_travel_payu_settings']['payment_option_payu'];
		}


		/**
		 * If current booking uses PayU.
		 *
		 * @return boolean
		 */
		public static function uses_payu_checkout() {
			$settings       = wptravel_get_settings();
			if( isset( $settings['selected_booking_option'] ) &&  count( $settings['selected_booking_option'] ) == 1 && $settings['selected_booking_option'][0] = 'booking-with-payment' ){ 
				return 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'payu' === $_REQUEST['wp_travel_payment_gateway'];
			}else{
				return 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_booking_option', $_REQUEST ) && 'booking_with_payment' === $_REQUEST['wp_travel_booking_option'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'payu' === $_REQUEST['wp_travel_payment_gateway'];
			}
			
		}

		/**
		 * Booking Message After booking.
		 *
		 * @return string
		 */
		public static function booking_message() {
			unset( $_SESSION['used-payu'] );
			$message = esc_html__( "We've received your booking and payment details. We'll contact you soon.", 'wp-travel-pro' );
			return $message;
		}

		/**
		 * Register/Enqueue Scripts.
		 */
		public static function assets() {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			if ( function_exists( 'wptravel_can_load_payment_scripts' ) ) {
				$load_payment_script = wptravel_can_load_payment_scripts();
			} else {
				$load_payment_script = wp_travel_can_load_payment_scripts();
			}
			if ( $load_payment_script && self::is_enabled() ) {
				wp_enqueue_script( 'payu-view-js', plugin_dir_url( __FILE__ ) . 'assets/js/wp-travel-payu-checkout-view' . $suffix . '.js', array( 'jquery' ), '1.0.0', true );
			}
		}

		/**
		 * Admin scripts.
		 */
		public static function admin_assets() {
			$screen = get_current_screen();
			if (  method_exists( 'WP_Travel',  'is_page' )  ) { // @since WP Travel 4.4.2
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_PAYU_CHECKOUT_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
			}

		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}


	}
endif;
