<?php
/**
 * PayU Payment Class.
 *
 * @package WP_Travel_PayU_Checkout_Core
 */

if ( ! class_exists( 'WP_Travel_PayU_Payment' ) ) :

	/**
	 * PayU Payment Handler.
	 */
	class WP_Travel_PayU_Payment {

		/**
		 * Defaults Args.
		 *
		 * @var $defaults
		 */
		private $defaults_vars;

		/**
		 * Holds Booking Id.
		 *
		 * @var $booking_id
		 */
		private $booking_id;

		/**
		 * Complete Partial Payment.
		 *
		 * @var $is_complete_partial_payment
		 */
		private $is_complete_partial_payment;

		/**
		 * Constructor.
		 *
		 * @param int  $booking_id Booking ID.
		 * @param bool $complete_partial_payment Is Complete Partial Payment.
		 */
		public function __construct( $booking_id = null, $complete_partial_payment = false ) {
			$this->booking_id                  = $booking_id;
			$this->is_complete_partial_payment = $complete_partial_payment;
			$this->defaults_vars               = array();
		}

		/**
		 * Get Return URL.
		 */
		public function get_return_url() {
			$trip_id = get_post_meta( (int) $this->booking_id, 'wp_travel_post_id', true );
			$url     = '';
			if ( function_exists( 'wp_travel_thankyou_page_url' ) ) {
				$url = wptravel_thankyou_page_url( (int) $trip_id );
			}
			return $url;
		}

		/**
		 * Merchant's Details Vars.
		 *
		 * @return array
		 */
		private function get_merchant_details() {
			$args     = array();
			$settings = $this->get_payment_settings();
			extract( $settings );
			$return_url = $this->get_return_url();
			// Merchant's Details.
			$args['merchant_id'] = $merchant_id;
			$args['secret']      = $merchant_secret;
			$args['return_url']  = add_query_arg(
				array(
					'booking_id' => $this->booking_id,
					'booked'     => true,
					'status'     => 'success',
					'order_id'   => $this->booking_id,
				),
				$return_url
			);
			$args['cancel_url']  = add_query_arg(
				array(
					'booking_id' => $this->booking_id,
					'booked'     => true,
					'status'     => 'cancel',
				),
				$return_url
			);
			$args['notify_url']  = esc_url( add_query_arg( 'PayU_listener', 'IPN', home_url( 'index.php' ) ) );
			return $args;
		}

		/**
		 * Buyer's Details Vars.
		 */
		public function get_buyer_details() {
			$booking_data = get_post_meta( (int) $this->booking_id, 'order_data', true );

			$args = array();
			$key  = $this->get_form_index_key();
			if ( $key ) :
				// Buyer Details.
				$args['first_name'] = isset( $booking_data['wp_travel_fname_traveller'][ $key ][0] ) ? $booking_data['wp_travel_fname_traveller'][ $key ][0] : '';
				$args['last_name']  = isset( $booking_data['wp_travel_lname_traveller'][ $key ][0] ) ? $booking_data['wp_travel_lname_traveller'][ $key ][0] : '';
				$args['email']      = isset( $booking_data['wp_travel_email_traveller'][ $key ][0] ) ? $booking_data['wp_travel_email_traveller'][ $key ][0] : '';
				$args['phone']      = isset( $booking_data['wp_travel_phone_traveller'][ $key ][0] ) ? $booking_data['wp_travel_phone_traveller'][ $key ][0] : '';
				$args['address']    = isset( $booking_data['wp_travel_address'] ) ? $booking_data['wp_travel_address'] : '';
				$args['city']       = isset( $booking_data['billing_city'] ) ? $booking_data['billing_city'] : '';
				$args['country']    = isset( $booking_data['wp_travel_country'] ) ? $booking_data['wp_travel_country'] : '';
			endif;
			return $args;
		}

		/**
		 * Transaction Details.
		 */
		public function get_transaction_details() {
			global $wt_cart;
			$total_price = $wt_cart->get_total();

			$arg             = array();
			$arg['order_id'] = uniqid( '', true );
			$trip_id         = get_post_meta( (int) $this->booking_id, 'wp_travel_post_id', true );
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			if ( $this->is_complete_partial_payment ) {
				$amount        = isset( $_POST['amount'] ) ? wp_unslash( $_POST['amount'] ) : 0;
				$arg['amount'] = $amount;
			} else {
				if ( isset( $_POST['wp_travel_payment_mode'] ) && 'partial' === $_POST['wp_travel_payment_mode'] ) {
					$arg['amount'] = $total_price['total_partial'];
				} else {
					$arg['amount'] = $total_price['total'];
				}
			}
			$arg['currency'] = isset( $settings['currency'] ) ? $settings['currency'] : '';
			$arg['items']    = get_the_title( (int) $trip_id );

			return $arg;
		}

		/**
		 * Checkout Form Index Key.
		 */
		private function get_form_index_key() {
			$order_details = get_post_meta( (int) $this->booking_id, 'order_items_data', true ); // Multiple Trips.

			$index = is_array( $order_details ) && count( $order_details ) > 0 ? array_keys( $order_details )[0] : null;
			return $index;
		}

		/**
		 * Payment Settings.
		 *
		 * @return array
		 */
		private function get_payment_settings() {
			$settings_instance = WP_Travel_PayU_Settings::instance();
			$settings          = $settings_instance->get_settings();
			return $settings;
		}
	}

endif;
