<?php
/**
 * WP Travel Partial Payment Core Class.
 *
 * @package wp-travel-partial-payment-core
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Partial Payment Core.
if ( ! class_exists( 'WP_Travel_Partial_Payment_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Partial_Payment_Core {

		const WP_TRAVEL_PARTIAL_PAYMENT_HANDLE = 'wp_travel_partial_payment_';

		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Partial Payment Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Partial_Payment_Core Instance.
		 * Ensures only one instance of WP_Travel_Partial_Payment_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Partial_Payment_Core()
		 * @return WP_Travel_Partial_Payment_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];
			self::includes();
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'admin_assets' ) );
		}

		/**
		 * Frontend assets.
		 */
		public static function frontend_assets() {

			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

			// Style.
			wp_enqueue_style( 'partial-style', self::$plugin_url . 'assets/css/custom' . $suffix . '.css', '', self::$version, true );

			// Scripts.
			wp_register_script( 'wp-travel-partial-payment-script', self::$plugin_url . 'assets/js/wp-travel-partial-payment' . $suffix . '.js', array( 'jquery' ), self::$version, true );
			wp_enqueue_script( 'wp-travel-partial-payment-script' );
		}

		/**
		 * Admin assets.
		 */
		public static function admin_assets() {
			$screen = get_current_screen();
			// @since WP Travel 4.4.2
			if (  method_exists( 'WP_Travel',  'is_page' )  ) {
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_PARTIAL_PAYMENT_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
			}
			wp_enqueue_script( 'wp-travel-partial-payment-admin-scripts', self::$plugin_url . 'assets/js/admin-scripts.js', array( 'jquery' ), self::$version, true );
		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}
			/**
			 * Include required files
			 */
			include sprintf( '%s/inc/functions.php', self::$abspath );
			// include sprintf( '%s/inc/class-frontend-assets.php', self::$abspath );
			include sprintf( '%s/inc/email-functions.php', self::$abspath );

			if ( self::is_request( 'admin' ) ) {
				include sprintf( '%s/inc/admin/settings.php', self::$abspath );
				include sprintf( '%s/inc/admin/class-multiple-partial-payments.php', self::$abspath ); // since 1.0.6
				
				include sprintf( '%s/core/helpers/settings.php', self::$abspath );
			}
			include sprintf( '%s/inc/admin/class-single-itinerary-price-tab.php', self::$abspath ); // since 1.0.6
			include sprintf( '%s/inc/class-frontend-process.php', self::$abspath );
			include sprintf( '%s/inc/class-payment-reminder-email.php', self::$abspath );
			
		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}


	}
endif;
