<?php
/**
 * This file has all the requried code for the enabling option for custom multiple partial payments in trip backend.
 *
 * @since 1.0.6
 * @package partial-payment-core/inc/admin/
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_Travel_Custom_Multiple_Partial_Payment' ) ) {
	/**
	 * Enables the option for the custom multiple partial payment in single itineray backend.
	 */
	class WP_Travel_Custom_Multiple_Partial_Payment {

		/**
		 * Init class.
		 */
		public static function init() {
			add_action( 'wp_travel_itinerary_price_tab_table_last_row', array( __CLASS__, 'hook_fields' ), 8 );
			add_filter( 'wp_travel_custom_partial_payment_string', array( __CLASS__, 'custom_partial_payment_string' ) );
		}

		/**
		 * Formats the partial payout percent.
		 *
		 * @param  int  $payout_percent
		 * @return void
		 */
		private static function formated_partial_payout( $payout_percent ) {
			if ( empty( $payout_percent ) ) {
				$payout_percent = 0;
			}
			if ( ! is_float( $payout_percent ) ) {
				$payout_percent = (float) $payout_percent;
			}
			return number_format( $payout_percent, 2, '.', '' );
		}

		/**
		 * Generates the custom partial payout fields according to the global fields or custom fields if custom payout has been set.
		 *
		 * @param array $custom_partial_payouts Custom added partial payment payouts percentage.
		 * @return void
		 */
		public static function get_partial_payout_fields( $trip_id ) {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$default_payout_percents = ! empty( $settings['minimum_partial_payout'] ) ? $settings['minimum_partial_payout'] : array();
			$initial_partial_payout  = wptravel_initial_partial_payout_unformated( $default_payout_percents );
			$custom_partial_payouts  = get_post_meta( $trip_id, 'wp_travel_minimum_partial_payout_percent', true );
			// Quick fix for the issue of generating false row.
			if ( is_array( $default_payout_percents ) && isset( $default_payout_percents[0] ) ) {
				unset( $default_payout_percents[0] );
			}
			if ( is_array( $default_payout_percents ) && count( $default_payout_percents ) > 0 ) {
				foreach ( $default_payout_percents as $key => $default_payout_percent ) {
					$payout_index                  = $key + 1;
					$partial_payouts               = isset( $custom_partial_payouts[ $key ] ) && ! empty( $custom_partial_payouts[ $key ] ) ? $custom_partial_payouts[ $key ] : $default_payout_percent;
					$partial_payouts               = self::formated_partial_payout( $partial_payouts );
					$custom_partial_payment_string = 'Custom Partial Payout ' . $payout_index . ' (%)';
					?>
					<tr class="multiple-custom-partial-payment-row">
						<td>
							<label for="wp-travel-minimum-partial-payout-percent-<?php echo esc_attr( $key ); ?>"><?php echo esc_html( $custom_partial_payment_string ); ?></label>
						</td>
						<td>
							<input type="number" min="1" max="100" step="0.01" name="wp_travel_minimum_partial_payout_percent[]" id="wp-travel-minimum-partial-payout-percent-<?php echo esc_attr( $key ); ?>" value="<?php echo esc_attr( $partial_payouts ); ?>">
							<p class="description"><?php echo esc_html__( 'Global partial payout: ', 'wp-travel-pro' ) . esc_html( $default_payout_percent ) . esc_html( '%' ); ?></p>
						</td>
					</tr>
					<?php
				}
			}
			self::warning_message( $trip_id );
		}

		/**
		 * Hook the generated fields to price tab backend.
		 *
		 * @param  int  $trip_id Trip ID.
		 * @return void
		 */
		public static function hook_fields( $trip_id ) {
			return self::get_partial_payout_fields( $trip_id );
		}

		/**
		 * Changes the Custom Minimum Payout text.
		 *
		 * @return void
		 */
		public static function custom_partial_payment_string( $custom_partial_payout_string ) {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$default_payout_percents = ! empty( $settings['minimum_partial_payout'] ) ? $settings['minimum_partial_payout'] : array();
			return is_array( $default_payout_percents ) && count( $default_payout_percents ) > 1 ? __( 'Custom Partial Payout 1 (%)', 'wp-travel-pro' ) : $custom_partial_payout_string;
		}

		/**
		 * Calculates the total percent of payout.
		 *
		 * @param  int  $trip_id
		 * @return void
		 */
		public static function get_custom_total_payout( $trip_id ) {
			$total_payout_percent = 0;
			$payout_percentages   = get_post_meta( $trip_id, 'wp_travel_minimum_partial_payout_percent', true );
			if ( ! empty( $payout_percentages ) && is_array( $payout_percentages ) ) {
				foreach ( $payout_percentages as $payout_percent ) {
					if ( ! empty( $payout_percent ) ) {
						$total_payout_percent += $payout_percent;
					}
				}
			}
			$total_percent = $total_payout_percent;
			return $total_percent;
		}

		/**
		 * Checks if custom payout is valid.
		 *
		 * @param  int     $trip_id
		 * @return boolean
		 */
		public static function is_custom_payout_valid( $trip_id ) {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$global_payouts       = isset( $settings['minimum_partial_payout'] ) && ! empty( $settings['minimum_partial_payout'] ) ? $settings['minimum_partial_payout'] : array();
			$count_global_payouts = is_array( $global_payouts ) ? count( $global_payouts ) : 1;
			$custom_total_payout  = self::get_custom_total_payout( $trip_id );
			/**
			 * Return true if no partial payouts has been added.
			 */
			if ( 1 === $count_global_payouts || 0 === $custom_total_payout ) {
				return true;
			}
			return $count_global_payouts > 0 && '100' === (string) $custom_total_payout;
		}

		/**
		 * Displays the warning message if payout is not equals to 100%.
		 *
		 * @return void
		 */
		public static function warning_message( $trip_id ) {
			$is_payouts_valid = self::is_custom_payout_valid( $trip_id );
			if ( ! $is_payouts_valid ) {
				?>
				<tr class="multiple-custom-partial-payment-row">
					<th></th>
					<td>
						<p class="wp-travel-notification notification-warning notice notice-error">
							<?php echo esc_html( 'Error: Total payout percent is not equals to 100%.' ); ?><br>
							<?php echo esc_html( 'Please update the trip once else global partial percent will be used as default.' ); ?>
						</p>
					</td>
				</tr>
				<?php
			}
		}
	}
}

WP_Travel_Custom_Multiple_Partial_Payment::init();
