<?php

/**
 * Multiple currency core file.
 *
 * @package multiple-currency-core
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_Travel_Multiple_Currency_Core' ) ) :
	class WP_Travel_Multiple_Currency_Core {
		protected static $abspath;

		protected static $plugin_file;

		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Multiple Currency Core
		 * @since 1.0.0
		 */
		protected static $instance = null;

		/**
		 * Main WP_Travel_Multiple_Currency_Core Instance.
		 * Ensures only one instance of WP_Travel_Multiple_Currency_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Multiple_Currency_Core()
		 * @return WP_Travel_Multiple_Currency_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		/**
		 * Init core.
		 */
		public static function init() {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$version     = '1.0.0';

			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Multiple_Currency_Core', 'wp_travel_multiple_currency_admin_styles' ) );
			self::wp_travel_multiple_currency_required_files();
			add_filter( 'wp_head', array( __CLASS__, 'multiple_currency_geolocation_data' ) );
			add_action( 'wp_enqueue_scripts', array( 'WP_Travel_Multiple_Currency_Core', 'wp_travel_multiple_currency_frontend_styles' ) );
		}

		/**
		 * Enqueue style and scripts for admin panel.
		 */
		public static function wp_travel_multiple_currency_admin_styles() {
			$localized = self::localized();
			$screen    = get_current_screen();
			if ( method_exists( 'WP_Travel', 'is_page' ) ) { // @since WP Travel 4.4.2
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_register_script( 'wp-travel-multiple-currency-admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
				wp_localize_script( 'wp-travel-multiple-currency-admin-settings', 'wp_travel_multiple_currency', $localized );
				wp_enqueue_script( 'wp-travel-multiple-currency-admin-settings' );
			}
			// Styles.
			wp_enqueue_style( 'select2-style' );
			// wp_enqueue_style( 'wp_travel_multiple_currency_admin_styles', plugin_dir_url( __FILE__ ) . 'assets/admin-styles.css', array(), '1.0.0' );

			// Scripts.
			wp_enqueue_script( 'select2-js' );
			wp_enqueue_script( 'wp_travel_multiple_currency_admin_scripts', plugin_dir_url( __FILE__ ) . 'assets/admin-scripts.js', array(), '1.0.0', true );
		}

		/**
		 * Localization.
		 */
		private static function localized() {

			$supported_currencies = array();
			$menu_locations       = array();

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$base_currency = isset( $settings['currency'] ) && ! empty( $settings['currency'] ) ? $settings['currency'] : '';
			if ( ! function_exists( 'wp_travel_multiple_currency_currencies_name' ) ) {
				return array(); // Quick fix. fatal error if multiple currency disabled.
			}
			$currencies      = wp_travel_multiple_currency_currencies_name();
			$theme_locations = WP_Travel_Multiple_Currency_Selector_Dropdown::get_theme_locations();

			if ( is_array( $currencies ) && count( $currencies ) > 0 ) {
				foreach ( $currencies as $currency_code => $currency_names ) {
					if ( function_exists( 'wptravel_get_currency_symbol' ) ) {
						$currency_symbols = wptravel_get_currency_symbol( $currency_code );
					} else {
						$currency_symbols = wp_travel_get_currency_symbol( $currency_code );
					}
					if ( ! empty( $currency_symbols ) && 'N/A' !== $currency_symbols ) {
						if ( 'BTC' !== $currency_code && $base_currency !== $currency_code ) { // Do not display the selected base currency in the selection list.
							$supported_currencies[] = array(
								'value' => $currency_code,
								'label' => html_entity_decode( "{$currency_names} ({$currency_symbols} - {$currency_code})" ),
							);
						}
					}
				}
			}

			if ( is_array( $theme_locations ) && ! empty( $theme_locations ) ) {
				foreach ( $theme_locations as $location_id => $location_label ) {
					$menu_locations[] = array(
						'value' => $location_id,
						'label' => $location_label,
					);
				}
			}

			return array(
				'currencies'           => $currencies,
				'server_status'        => $_SERVER,
				'menu_locations'       => $menu_locations,
				'supported_currencies' => $supported_currencies,
			);
		}

		/**
		 * Enqueues our external stylesheet for frontend
		 */
		public static function wp_travel_multiple_currency_frontend_styles() {

			// Scripts.
			wp_enqueue_script( 'wp_travel_multiple_currency_frontend_scripts', plugin_dir_url( __FILE__ ) . 'assets/scripts.js', array(), '1.0.0', true );
			wp_localize_script( 
				'wp_travel_multiple_currency_frontend_scripts', 
				'wp_travel_multiple_currency', 
				array(
					'geo_location_conformation' => apply_filters( 'wptravel_geo_location_confirmation', false )
				)
			);
		}

		/**
		 * Hook geolocation data to wp_head.
		 * Then use that data from javascript to set the cookie.
		 * After the cookie is set, the remaining process is carried on the same way as the currency selector dropdown.
		 *
		 * @todo Do the below work by localizing the data. Use the WP Travel default localize filter.
		 */
		public static function multiple_currency_geolocation_data() {
			$geo_location_data = WP_Travel_Multiple_Currency_API_Data::geolocation_data();
			$currency_code     = isset( $geo_location_data['currency'] ) && ! empty( $geo_location_data['currency'] ) ? $geo_location_data['currency'] : '';
			$use_geolocation   = wp_travel_multiple_currency_use_geo_location();
			$set_geolocation   = ! isset( $_COOKIE['wp_travel_multiple_currency_customerSelectedCurrency'] ) || empty( $_COOKIE['wp_travel_multiple_currency_customerSelectedCurrency'] ) ? true : false;
			if ( $use_geolocation ) {
				if ( $set_geolocation ) {
					?>
					<span id="use_geolocation" data-use-geolocation="<?php echo esc_attr( $use_geolocation ); ?>"></span>
					<span id="geolocation_currency_code" data-geolocation-currency-code="<?php echo esc_attr( $currency_code ); ?>"></span>
					<?php
				}
			}
		}

		/**
		 * Function to include files after plugin loaded.
		 */
		public static function wp_travel_multiple_currency_required_files() {
			include_once self::$abspath . '/inc/functions.php';
			include_once self::$abspath . '/inc/hooks.php';
			include_once self::$abspath . '/inc/class-process-api-data.php';
			include_once self::$abspath . '/inc/class-transient-api-data.php';
			include_once self::$abspath . '/inc/class-admin-settings-misc-options.php';
			include_once self::$abspath . '/inc/class-api-base-currency.php';
			include_once self::$abspath . '/inc/class-currency-selector-dropdown.php';
			include_once self::$abspath . '/inc/multiple-currency-ajax.php';
		}
	}
endif;
