<?php
/**
 * @package multiple-currency-core/inc/
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_Travel_Multiple_Currency_Selector_Dropdown' ) ) {

	/**
	 * Class that init the currency selector at front end.
	 */
	class WP_Travel_Multiple_Currency_Selector_Dropdown {

		/**
		 * Initialize selector.
		 *
		 * @return void
		 */
		public static function init() {
			self::enable_selector();
			add_filter( 'wp_nav_menu_items', array( __CLASS__, 'selector_dropdown' ), 10, 2 );
		}

		/**
		 * Return the navigation display locations.
		 *
		 * @return array $display_locations.
		 */
		public static function get_theme_locations() {
			$display_locations   = array();
			$menus               = get_nav_menu_locations();
			$menu_location_slugs = array_keys( $menus );
			if ( is_array( $menus ) && count( $menus ) > 0 ) {
				foreach ( $menu_location_slugs as $menu_location_slug ) {
					$menu_location_title                      = ucfirst( str_replace( '-', ' ', $menu_location_slug ) );
					$display_locations[ $menu_location_slug ] = $menu_location_title;
				}
			}
			return $display_locations;
		}

		/**
		 * Displays the currency selector dropdown at frontend in selected theme location.
		 *
		 * @param  string $items
		 * @param  object $args
		 * @return string $items
		 */
		public static function selector_dropdown( $items = '', $args ) {

			if ( ! self::enable_selector() ) {
				return $items;
			}

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$new_exchange_rates           = WP_Travel_Multiple_Currency_Base_Currency::exchange_rates();
			$base_currency                = ! empty( $new_exchange_rates['base'] ) ? $new_exchange_rates['base'] : '';
			$display_selector_at          = ! empty( $settings['display_currency_selector_at'] ) && isset( $settings['display_currency_selector_at'] ) ? $settings['display_currency_selector_at'] : '';
			$selected_multiple_currencies = ! empty( $settings['global_selected_multiple_currencies'] ) && isset( $settings['global_selected_multiple_currencies'] ) ? $settings['global_selected_multiple_currencies'] : array();
			$customer_selected_currency   = self::get_customer_selected_currency();
			if ( $args->theme_location === $display_selector_at ) {
				if ( is_array( $selected_multiple_currencies ) && count( $selected_multiple_currencies ) > 0 ) {
					ob_start();
					?>
					<li class="wp-travel-multiple-currency currency-dropdown">
						<form method="post">
							<select name="customer_selected_currency" id="customer_selected_currency">
								<option value="<?php echo esc_attr( $base_currency ); ?>">
									<?php echo esc_html( wp_travel_multiple_currency_parse_symbols( $base_currency ) ); ?>
								</option>
								<?php
								foreach ( $selected_multiple_currencies as $backend_selected_currency ) {
									if ( ! empty( $backend_selected_currency ) ) {
										if ( $base_currency !== $backend_selected_currency ) {
											?>
											<option <?php selected( $customer_selected_currency, $backend_selected_currency ); ?> value="<?php echo esc_attr( $backend_selected_currency ); ?>">
												<?php echo esc_html( wp_travel_multiple_currency_parse_symbols( $backend_selected_currency ) ); ?>
											</option>
											<?php
										}
									}
								}
								?>
							</select>
						</form>
					</li>
					<?php
					$items .= ob_get_clean();
				}
			}

			return $items;
		}

		/**
		 * Returns the cookie data that has customer selected currency.
		 *
		 * @return array
		 */
		public static function get_customer_selected_currency() {
			$trip_id       = get_the_ID();
			$base_currency = WP_Travel_Multiple_Currency_Base_Currency::get_selected_currency();
			$currency_code = isset( $_COOKIE['wp_travel_multiple_currency_customerSelectedCurrency'] ) && ! empty( $_COOKIE['wp_travel_multiple_currency_customerSelectedCurrency'] ) ? sanitize_text_field( wp_unslash( $_COOKIE['wp_travel_multiple_currency_customerSelectedCurrency'] ) ) : $base_currency;

			/**
			 * Supports for custom multiple currency according to the trip id.
			 *
			 * @since wp travel pro 2.1.3
			 */
			return apply_filters( 'wp_travel_multiple_currency_user_selected_currency', $currency_code, $trip_id );
		}

		/**
		 * Returns false if the current page is cart, checkout or dashboard page.
		 *
		 * @return bool
		 */
		public static function enable_selector() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$multiple_currency_api_key = ! empty( $settings['exchange_rate_api_key'] ) && isset( $settings['exchange_rate_api_key'] ) ? $settings['exchange_rate_api_key'] : '';
			$use_geo_location          = wp_travel_multiple_currency_use_geo_location();
			if ( empty( $multiple_currency_api_key ) ) {
				return false;
			}
			if ( $use_geo_location ) {
				return false;
			}

			if ( method_exists( 'WP_Travel', 'is_page' ) ) { // @since WP Travel 4.4.2
				if ( WP_Travel::is_page( 'cart' ) || WP_Travel::is_page( 'checkout' ) || WP_Travel::is_page( 'dashboard' ) ) {
					return false;
				}
			} else {
				if ( wp_travel_is_cart_page() || wp_travel_is_checkout_page() || wp_travel_is_dashboard_page() ) {
					return false;
				}
			}
			return apply_filters( 'wp_travel_multiple_currency_enable_selector', true ); // since wp travel pro v2.1.3.
		}
	}
}
add_action( 'init', array( 'WP_Travel_Multiple_Currency_Selector_Dropdown', 'init' ) );
// WP_Travel_Multiple_Currency_Selector_Dropdown::init();


function WP_Travel_Multiple_Currency_Selector_Dropdown_Shortcode(){
	if ( ! WP_Travel_Multiple_Currency_Selector_Dropdown::enable_selector() ) {
		return $items;
	}

	if ( function_exists( 'wptravel_get_settings' ) ) {
		$settings = wptravel_get_settings();
	} else {
		$settings = wp_travel_get_settings();
	}
	$new_exchange_rates           = WP_Travel_Multiple_Currency_Base_Currency::exchange_rates();
	$base_currency                = ! empty( $new_exchange_rates['base'] ) ? $new_exchange_rates['base'] : '';
	$display_selector_at          = ! empty( $settings['display_currency_selector_at'] ) && isset( $settings['display_currency_selector_at'] ) ? $settings['display_currency_selector_at'] : '';
	$selected_multiple_currencies = ! empty( $settings['global_selected_multiple_currencies'] ) && isset( $settings['global_selected_multiple_currencies'] ) ? $settings['global_selected_multiple_currencies'] : array();
	$customer_selected_currency   = WP_Travel_Multiple_Currency_Selector_Dropdown::get_customer_selected_currency();

		if ( is_array( $selected_multiple_currencies ) && count( $selected_multiple_currencies ) > 0 ) {
			ob_start();
			?>
			<div class="wp-travel-multiple-currency currency-dropdown">
				<form method="post">
					<select name="customer_selected_currency" id="customer_selected_currency">
						<option value="<?php echo esc_attr( $base_currency ); ?>">
							<?php echo esc_html( wp_travel_multiple_currency_parse_symbols( $base_currency ) ); ?>
						</option>
						<?php
						foreach ( $selected_multiple_currencies as $backend_selected_currency ) {
							if ( ! empty( $backend_selected_currency ) ) {
								if ( $base_currency !== $backend_selected_currency ) {
									?>
									<option <?php selected( $customer_selected_currency, $backend_selected_currency ); ?> value="<?php echo esc_attr( $backend_selected_currency ); ?>">
										<?php echo esc_html( wp_travel_multiple_currency_parse_symbols( $backend_selected_currency ) ); ?>
									</option>
									<?php
								}
							}
						}
						?>
					</select>
				</form>
			</div>
			<?php
			$items .= ob_get_clean();
		}


	return $items;
}

add_shortcode( 'Multiple_Currency_Selector', 'WP_Travel_Multiple_Currency_Selector_Dropdown_Shortcode' );

