<?php
/**
 * @package multiple-currency-core/inc/
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_Travel_Multiple_Currency_Settings' ) ) {
	/**
	 * Class which has all the required settings.
	 */
	class WP_Travel_Multiple_Currency_Settings {

		/**
		 * Create single instance.
		 */
		private static $instance = null;

		/**
		 * Get Instance.
		 *
		 * @return instance $instance Instance.
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		/**
		 * Initialize WP_Travel_Multiple_Currency_Settings.
		 */
		public static function init() {
			self::$instance = __CLASS__;
			if ( ! ( WP_Travel_Multiple_Currency_API_Data::is_active_currency_exchange_rates() ) ) {
				add_filter( 'wp_travel_settings_fields', array( __CLASS__, 'api_key_fields' ) );
				add_action( 'wp_travel_tabs_content_settings_misc_options_global', array( __CLASS__, 'api_key_fields_callback' ), 20, 2 );
			}
			add_filter( 'wp_travel_settings_fields', array( __CLASS__, 'settings_fields' ) );
			add_action( 'wp_travel_tabs_content_settings_misc_options_global', array( __CLASS__, 'settings_fields_callback' ), 20, 2 );

			// Purge transient if "Reset" button is clicked.
			WP_Travel_Multiple_Currency_API_Transient::transient_reset();
		}

		/**
		 * Add settings field is currency exchange rate is not already activate.
		 *
		 * @param  array $fields Settings array fields.
		 * @return array $fields Settings array new added fields
		 */
		public static function api_key_fields( $fields ) {
			$fields['exchange_rate_api_key']                  = '';
			$fields['exchange_rate_api_key_premium_checkbox'] = '';
			return $fields;
		}

		/**
		 * Add default settings field.
		 *
		 * @param  array $fields Settings array fields.
		 * @return array $fields Settings array new added fields
		 */
		public static function settings_fields( $fields ) {
			$fields['global_selected_multiple_currencies'] = array();
			$fields['display_currency_selector_at']        = '';
			$fields['use_custom_conversion_rates']         = '';
			$fields['enable_geo_location']                 = '';
			$fields['custom_conversions_values']           = array();
			$fields['show_wp_travel_multiple_currency']    = 'no'; // Default disabled.

			return $fields;
		}

		/**
		 * Callback function which will have input fields.
		 *
		 * @param string $tab Tab name.
		 * @param array  $args Settings data arguments.
		 * @return void
		 */
		public static function api_key_fields_callback( $tab, $args ) {

			if ( 'misc_options_global' !== $tab ) {
				return;
			}
			$settings                         = ! empty( $args ) ? $args['settings'] : '';
			$multiple_currency_api_key        = ! empty( $settings['exchange_rate_api_key'] ) && isset( $settings['exchange_rate_api_key'] ) ? $settings['exchange_rate_api_key'] : '';
			$multiple_currency_api_is_premium = ! empty( $settings['exchange_rate_api_key_premium_checkbox'] ) && isset( $settings['exchange_rate_api_key_premium_checkbox'] ) ? $settings['exchange_rate_api_key_premium_checkbox'] : '';
			?>
			<table class="form-table">
			<h3 class="wp-travel-tab-content-title"><?php echo esc_html__( 'Fixer API', 'wp-travel-pro' ); ?></h3>
				<tbody>
					<tr>
						<th>
							<label for="multiple_currency_api_key"><?php echo esc_html__( 'Enter Your API Key:', 'wp-travel-pro' ); ?></label>
						</th>
						<td>
							<input type="text" placeholder="<?php echo esc_attr__( 'Your API Access Key', 'wp-travel-pro' ); ?>" id="multiple_currency_api_key" name="exchange_rate_api_key" value="<?php echo esc_html( $multiple_currency_api_key ); ?>"/>
							<?php // phpcs:disable ?>
							<?php echo WP_Travel_Multiple_Currency_API_Data::get_description(); ?>
							<?php // phpcs:enable ?>
						</td>
					</tr>
					<tr>
						<th>
							<label for="multiple_currency_api_is_premium"><?php echo esc_html__( 'Premium API Key Subscription:', 'wp-travel-pro' ); ?></label>
						</th>
						<td>
							<span class="show-in-frontend checkbox-default-design">
								<label data-on="Yes" data-off="No">
									<input value="no" name="exchange_rate_api_key_premium_checkbox" type="hidden">
									<input type="checkbox" id="multiple_currency_api_is_premium" name="exchange_rate_api_key_premium_checkbox" value="yes" <?php checked( $multiple_currency_api_is_premium, 'yes' ); ?>/>
									<span class="switch"></span>
								</label>
							</span>
							<p class="description"><?php echo esc_html__( 'Toggle it on if you have premium ( HTTPS Encryption ) API key subscription.', 'wp-travel-pro' ); ?></p>
						</td>
					</tr>
				</tbody>
			</table>
			<?php
		}

		/**
		 * Callback function which will have input fields.
		 *
		 * @param string $tab Tab name.
		 * @param array  $args Settings data arguments.
		 * @return void
		 */
		public static function settings_fields_callback( $tab, $args ) {
			if ( 'misc_options_global' !== $tab ) {
				return;
			}
			$settings                     = ! empty( $args ) ? $args['settings'] : '';
			$supported_currencies         = wp_travel_multiple_currency_currencies_name();
			$use_custom_conversion_rates  = ! empty( $settings['use_custom_conversion_rates'] ) && isset( $settings['use_custom_conversion_rates'] ) ? $settings['use_custom_conversion_rates'] : '';
			$selected_multiple_currencies = ! empty( $settings['global_selected_multiple_currencies'] ) && isset( $settings['global_selected_multiple_currencies'] ) ? $settings['global_selected_multiple_currencies'] : array();
			$display_selector_at          = ! empty( $settings['display_currency_selector_at'] ) && isset( $settings['display_currency_selector_at'] ) ? $settings['display_currency_selector_at'] : '';
			$new_exchange_rates           = WP_Travel_Multiple_Currency_Base_Currency::exchange_rates();
			$theme_locations              = WP_Travel_Multiple_Currency_Selector_Dropdown::get_theme_locations();
			$base_currency                = ! empty( $new_exchange_rates['base'] ) ? $new_exchange_rates['base'] : 'N/A';

			// Filters.
			$enable_geo_location_module = apply_filters( 'wp_trave_multiple_currency_enable_geo_location_module', true );
			$enable_custom_rates_module = apply_filters( 'wp_trave_multiple_currency_enable_custom_rates_module', false );
			?>
			<table class="form-table">
			<h3 class="wp-travel-tab-content-title"><?php echo esc_html__( 'Multiple Currency', 'wp-travel-pro' ); ?></h3>
				<tbody>
					<?php if ( empty( $settings['exchange_rate_api_key'] ) ) { ?>
						<tr>
							<th>
							</th>
							<td class="wp-travel-notification notification-warning notice notice-error">
								<p><?php esc_html_e( 'API Status: Fixer API key invalid or empty.', 'wp-travel-pro' ); ?></p>
							</td>
						</tr>
					<?php } ?>
					<tr>
						<th>
							<label><?php esc_html_e( 'Base Currency:', 'wp-travel-pro' ); ?></label>
						</th>
						<td>
							<?php echo ( ( isset( $supported_currencies[ $base_currency ] ) ) ? esc_html( $supported_currencies[ $base_currency ] . ' ( ' . $base_currency . ' )' ) : 'N/A' ); ?>
						</td>
					</tr>
					<?php
					// Geo-location.
					if ( $enable_geo_location_module ) {
						self::geo_location_module( $settings );
					}
					?>
					<tr class="multiple-currency-tr-hidable">
						<th>
							<label><?php esc_html_e( 'Display location:', 'wp-travel-pro' ); ?></label>
						</th>
						<td>
							<select name="display_currency_selector_at" id="display_currency_selector_at">
								<?php
								if ( is_array( $theme_locations ) && count( $theme_locations ) > 0 ) {
									foreach ( $theme_locations as $theme_location_slug => $theme_location_title ) {
										?>
										<option <?php selected( $display_selector_at, $theme_location_slug ); ?> value="<?php echo esc_attr( $theme_location_slug ); ?>"><?php echo esc_html( $theme_location_title ); ?></option>
										<?php
									}
								}
								?>
							</select>
						</td>
					</tr>
					<tr>
						<th>
							<label for="global_selected_multiple_currencies"><?php echo esc_html__( 'Select Currencies:', 'wp-travel-pro' ); ?></label>
						</th>
						<td>
							<select name="global_selected_multiple_currencies[]" id="global_selected_multiple_currencies" multiple="multiple">
							<?php
							if ( is_array( $supported_currencies ) && count( $supported_currencies ) > 0 ) {
								foreach ( $supported_currencies as $currency_symbols => $currency_names ) {
									$selected       = in_array( $currency_symbols, $selected_multiple_currencies, true ) ? 'selected' : '';
									if ( function_exists( 'wptravel_get_currency_symbol' ) ) {
										$actual_symbols = wptravel_get_currency_symbol( $currency_symbols );
									} else {
										$actual_symbols = wp_travel_get_currency_symbol( $currency_symbols );
									}
									if ( 'N/A' !== $actual_symbols ) {
										if ( ! empty( $actual_symbols ) ) {
											if ( 'BTC' !== $currency_symbols ) {
												if ( $base_currency !== $currency_symbols ) { // Do not display the selected base currency in the selection list.
													?>
													<option <?php echo esc_attr( $selected ); ?> value="<?php echo esc_attr( $currency_symbols ); ?>"><?php echo esc_html( $currency_names ); ?> - ( <?php echo esc_html( $currency_symbols . ' - ' . $actual_symbols ); ?> )</option>
													<?php
												}
											}
										}
									}
								}
							}
							?>
							</select>
							<p class="description">
								<?php esc_html_e( 'Please only select those currencies which are supported by your payment gateway.', 'wp-travel-pro' ); ?>
							</p>
						</td>
					</tr>
					<?php
					// Custom rates.
					if ( $enable_custom_rates_module ) {
						self::custom_rate_module( $settings );
					}
					?>
					<tr>
						<th>
							<label for="multiple_currency_reset_transient"><?php echo esc_html__( 'Reset Transient Cache:', 'wp-travel-pro' ); ?></label>
						</th>
						<td>
							<input type="submit" class="multiple_currency_reset_transient" name="multiple_currency_reset_transient" value="<?php echo esc_html__( 'Reset', 'wp-travel-pro' ); ?>">
							<p class="description"><?php echo esc_html__( 'Cache automatically replaced with new data in every 4 hours, to force reset click "Reset" button.', 'wp-travel-pro' ); ?></p>
						</td>
					</tr>
				</tbody>
			</table>
			<?php
			/**
			 * Send empty value if none currencies selected.
			 */
			if ( ! isset( $_POST['global_selected_multiple_currencies'] ) ) { //phpcs:ignore
				?>
					<input type="hidden" name="global_selected_multiple_currencies[]" value=''>
				<?php
			}
		}

		/**
		 * Adds the toggle switch for the geo-location enable.
		 *
		 * @param array $settings Saved and default Settings.
		 * @return void
		 */
		public static function geo_location_module( $settings ) {
			$enable_geo_location = ! empty( $settings['enable_geo_location'] ) && isset( $settings['enable_geo_location'] ) ? $settings['enable_geo_location'] : '';
			?>
			<tr>
				<th>
					<label for="enable_geo_location"><?php esc_html_e( 'Enable Geo-Location:', 'wp-travel-pro' ); ?></label>
				</th>
				<td>
					<span class="show-in-frontend checkbox-default-design">
						<label data-on="Yes" data-off="No">
							<input value="no" name="enable_geo_location" type="hidden">
							<input type="checkbox" id="enable_geo_location" name="enable_geo_location" value="yes" <?php checked( $enable_geo_location, 'yes' ); ?>/>
							<span class="switch"></span>
						</label>
					</span>
					<p class="description"><?php echo esc_html__( 'Toggle it on to use the geo-location feature. Please note, you cannot use another manual currency selector option if toggled on.', 'wp-travel-pro' ); ?></p>
				</td>
			</tr>
			<?php
		}

		/**
		 * Adds the feature for the custom rates.
		 *
		 * @param array $settings Saved and default Settings.
		 * @return void
		 */
		public static function custom_rate_module( $settings ) {
			$supported_currencies         = wp_travel_multiple_currency_currencies_name();
			$selected_multiple_currencies = ! empty( $settings['global_selected_multiple_currencies'] ) && isset( $settings['global_selected_multiple_currencies'] ) ? $settings['global_selected_multiple_currencies'] : array();
			$use_custom_conversion_rates  = ! empty( $settings['use_custom_conversion_rates'] ) && isset( $settings['use_custom_conversion_rates'] ) ? $settings['use_custom_conversion_rates'] : '';
			$custom_added_rates           = ! empty( $settings['custom_conversions_values'] ) && isset( $settings['custom_conversions_values'] ) ? $settings['custom_conversions_values'] : array();
			$new_exchange_rates           = WP_Travel_Multiple_Currency_Base_Currency::exchange_rates();
			$base_currency                = ! empty( $new_exchange_rates['base'] ) ? $new_exchange_rates['base'] : 'N/A';
			?>
			<tr>
				<th>
					<label for="use_custom_conversion_rates"><?php echo esc_html__( 'Set Custom Conversion Rates?', 'wp-travel-pro' ); ?></label>
				</th>
				<td>
					<span class="show-in-frontend checkbox-default-design">
						<label data-on="Yes" data-off="No">
							<input value="no" name="use_custom_conversion_rates" type="hidden">
							<input type="checkbox" id="use_custom_conversion_rates" name="use_custom_conversion_rates" value="yes" <?php checked( $use_custom_conversion_rates, 'yes' ); ?>/>
							<span class="switch"></span>
						</label>
					</span>
					<p class="description"><?php echo esc_html__( 'Toggle it on if you want to set the custom conversion rates for selected currencies.', 'wp-travel-pro' ); ?></p>
				</td>
			</tr>
			<tr class="hide-this-tr-custom-currency-off">
				<th>
					<label for="custom_conversion_rates"><?php echo esc_html__( 'Custom Conversion Rates:', 'wp-travel-pro' ); ?></label>
				</th>
				<td>
					<table id="custom_conversion_rates_internal_table">
						<tr>
							<th><?php echo ( ( isset( $supported_currencies[ $base_currency ] ) ) ? esc_html( $supported_currencies[ $base_currency ] . ' (' . $base_currency . ' )' ) : 'N/A' ); ?></th>
							<th><?php esc_html_e( 'Conversion Currency', 'wp-travel-pro' ); ?></th>
							<th><?php esc_html_e( 'Symbols', 'wp-travel-pro' ); ?></th>
							<th><?php esc_html_e( 'Default Rates', 'wp-travel-pro' ); ?></th>
							<th><?php esc_html_e( 'Custom Rates', 'wp-travel-pro' ); ?></th>
						</tr>
						<?php
						if ( ! empty( $new_exchange_rates ) && count( $selected_multiple_currencies ) > 1 ) {
							foreach ( $selected_multiple_currencies as $selected_multiple_currency ) {
								$custom_rates  = ! empty( $custom_added_rates[ $selected_multiple_currency ] ) && isset( $custom_added_rates[ $selected_multiple_currency ] ) ? (float) $custom_added_rates[ $selected_multiple_currency ] : '';
								$custom_rates  = ( is_float( $custom_rates ) ) && ! empty( $custom_rates ) ? number_format( $custom_rates, 4, '.', '' ) : '';
								$default_rates = isset( $new_exchange_rates['rates'][ $selected_multiple_currency ] ) ? number_format( $new_exchange_rates['rates'][ $selected_multiple_currency ], 4, '.', '' ) : 0;
								if ( $default_rates > 0 ) {
									?>
									<tr id="<?php echo esc_html( $selected_multiple_currency ); ?>">
										<td>
											<?php echo esc_html( '1' ); // Base value ?>
										</td>
										<td>
											<label for="currency_<?php echo esc_html( $selected_multiple_currency ); ?>"><?php echo esc_html( $supported_currencies[ $selected_multiple_currency ] ); // Currency name ?></label>
										</td>
										<td>
											<?php echo esc_html( $selected_multiple_currency ); // Currency symbols ?>
										</td>
										<td>
											<span><?php echo esc_html( $default_rates ); // Defualt api value ?></span>
										</td>
										<td>
											<input id="currency_<?php echo esc_html( $selected_multiple_currency ); ?>" type="number" step="0.0001" placeholder="<?php esc_attr_e( 'Enter custom rates', 'wp-travel-pro' ); ?>" class="custom-conversion-rates" name="custom_conversions_values[<?php echo esc_html( $selected_multiple_currency ); ?>]" value="<?php echo esc_html( $custom_rates ); ?>">
										</td>
									</tr>
									<?php
								}
							}
						}
						?>
					</table>
				</td>
			</tr>
			<?php
		}
	}
}

/**
 * Initialize settings class.
 *
 * @return void
 */
function wp_travel_multiple_currency_init_settings() {
	WP_Travel_Multiple_Currency_Base_Currency::exchange_rates(); // Due to transient data not loading issue while saving.
	WP_Travel_Multiple_Currency_Settings::init();
}
add_action( 'init', 'wp_travel_multiple_currency_init_settings' );
