<?php
/**
 * WP Travel Invoice Core Class.
 *
 * @package wp-travel-invoice-core
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
// WP Travel Invoice core files.
if ( ! class_exists( 'WP_Travel_Invoice_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Invoice_Core {

		const WP_TRAVEL_INVOICE_HANDLE = 'wp_travel_invoice_';

		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Invoice Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Invoice_Core Instance.
		 * Ensures only one instance of WP_Travel_Invoice_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Invoice_Core()
		 * @return WP_Travel_Invoice_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			define( 'WP_TRAVEL_INVOICE_PATH', WP_CONTENT_DIR . '/' . 'wt-invoice' . '/' );
			define( 'WP_TRAVEL_INVOICE_TMP_PATH', WP_CONTENT_DIR . '/' . 'tmp' );
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];

			self::includes();

			add_filter( 'wp_travel_display_critical_admin_notices', array( 'WP_Travel_Invoice_Core', 'display_notices' ) );
			add_action( 'wp_travel_critical_admin_notice', array( 'WP_Travel_Invoice_Core', 'notices' ) );

			add_action( 'wp_travel_booking_metabox_after_title', array( 'WP_Travel_Invoice_Core', 'download_pdf' ) );
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'admin_assets' ) );
			add_filter( 'wp_travel_strings', array( __CLASS__, 'pro_invoice_strings' ) );
		}

		/**
		 * Utilities Frontend assets.
		 */
		public static function frontend_assets() {

			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		}

		/**
		 * string array create
		 */
		public static function pro_invoice_strings( $args ) {
			$args['invoice'] = [
				'particular' 	=> __( 'Particular', 'wp-travel-pro' ),
				'price'			=> __( 'Price', 'wp-travel-pro' ),
				'trip_pax'		=> __( 'Trip Pax', 'wp-travel-pro' ),
				'trip_amount'	=> __( 'Trip Amount', 'wp-travel-pro' ),
				'customer_name'			=> __( 'Customer Name:', 'wp-travel-pro' ),
				'customer_email'		=> __( 'Customer Email:', 'wp-travel-pro' ),
				'address'				=> __( 'Address:', 'wp-travel-pro' ),
				'phone'					=> __( 'Phone:', 'wp-travel-pro' ),
				'invoice_number'		=> __( 'Invoice Number:', 'wp-travel-pro' ),
				'invoice_date'			=> __( 'Date:', 'wp-travel-pro' )
			];
			return apply_filters( 'wp_travel_pro_invoice_strings', $args );
		}
		/**
		 * Utilities admin assets.
		 */
		public static function admin_assets() {
			$screen = get_current_screen();
			if ( method_exists( 'WP_Travel', 'is_page' ) ) { // @since WP Travel 4.4.2
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_INVOICE_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
			}
		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}

			include sprintf( '%s/inc/email-functions.php', self::$abspath );

			if ( self::is_request( 'frontend' ) ) {
				if ( function_exists( 'wptravel_get_settings' ) ) {
					$settings = wptravel_get_settings();
				} else {
					$settings = wp_travel_get_settings();
				}
				add_filter( 'wp_travel_email_template_type', array( __CLASS__, 'template_type' ), 10, 3 );
				// if ( 'yes' === @$settings['enable_invoice'] ) { // Send invoice if enabled.
					add_action( 'wp_travel_after_payment_email_sent', array( __CLASS__, 'send_invoice' ), 10, 3 );
				// }
			}

			if ( self::is_request( 'admin' ) ) {
				include sprintf( '%s/inc/admin/settings.php', self::$abspath );
			}
		}

		public static function template_type( $email_data, $type, $sent_to ) {
			if ( 'invoice' == $type ) {
				if ( function_exists( 'wptravel_get_settings' ) ) {
					$settings = wptravel_get_settings();
				} else {
					$settings = wp_travel_get_settings();
				}
				$invoice_client_template      = isset( $settings['invoice_client_template'] ) && is_array( $settings['invoice_client_template'] ) ? $settings['invoice_client_template'] : array();
				$email_data['client_subject'] = $invoice_client_template['client_subject'];
				$email_data['email_content']  = $invoice_client_template['email_content'];
			}
			return $email_data;
		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}

		public static function send_invoice( $booking_id, $email_data, $email_tags ) {
			if ( ! $booking_id || ! $email_data || ! $email_tags ) {
				return;
			}

			$last_invoice_id = get_option( 'wt_last_invoice_id' );

			if ( ! $last_invoice_id ) {
				$invioce_id = 1;
			} else {
				$invioce_id = (int) $last_invoice_id + 1;
			}
			update_post_meta( $booking_id, 'wt_invoice_id', $invioce_id );
			update_option( 'wt_last_invoice_id', $invioce_id );

			// For PDF.
			require sprintf( '%svendor/autoload.php', self::$abspath );

			$defaultConfig = ( new Mpdf\Config\ConfigVariables() )->getDefaults();
			$font_dirs     = $defaultConfig['fontDir'];

			$defaultFontConfig = ( new Mpdf\Config\FontVariables() )->getDefaults();
			$fontData          = $defaultFontConfig['fontdata'];

			$mpdf = new \Mpdf\Mpdf(
				array(
					'margin_top'    => 0,
					'margin_left'   => 0,
					'margin_bottom' => 0,
					'margin_right'  => 0,
					'tempDir'       => WP_TRAVEL_INVOICE_TMP_PATH,
					'fontDir'       => array_merge(
						$font_dirs,
						array(
							__DIR__ . '/fonts',
						)
					),
				)
			);

			ob_start();
			self::invoice_template( $booking_id );
			$html = ob_get_contents();
			ob_end_clean();
			/**
			 * @since 5.9
			 * fixed download using chinies lan
			 */
			$site_languages = get_locale();
			if ( $site_languages == 'zh_CN' || $site_languages == 'zh_TW' || $site_languages == 'ja' || $site_languages == 'zh_HK' ) {
				$mpdf->useAdobeCJK      = true;
				$mpdf->autoLangToFont   = true;
				$mpdf->autoScriptToLang = true;
			}
			$mpdf->AddPage();
			$mpdf->WriteHTML( $html );

			// $upload_dir = wp_upload_dir();
			$dir = trailingslashit( WP_TRAVEL_INVOICE_PATH );

			$filename = $invioce_id . '.pdf';

			$mpdf->Output( $dir . $filename, 'F' );  // Save pdf file in directory.
			$attachment = $dir . $filename; // Saved file name along with full path.

			// Email Starts.
			$email = new WP_Travel_Emails();

			// Admin Payment Email Vars.
			$invoice_email_template = $email->wptravel_get_email_template( 'invoice', 'client' );

			$client_message_data  = $invoice_email_template['mail_header'];
			$client_message_data .= $invoice_email_template['mail_content'];
			$client_message_data .= $invoice_email_template['mail_footer'];

			// Admin message.
			$client_invoice_message = str_replace( array_keys( $email_tags ), $email_tags, $client_message_data );
			// Admin Subject.
			$client_invoice_subject = $invoice_email_template['subject'];

			$from = $email_data['from'];
			$to   = $email_data['to'];

			// To send HTML mail, the Content-type header must be set.
			// $headers = $email->email_headers( $to, $from );

			$headers = array('Content-Type: text/html; charset=UTF-8');

			if ( ! wp_mail( $to, $client_invoice_subject, $client_invoice_message, $headers, array( $attachment ) ) ) {
				WP_Travel()->notices->add( '<strong>' . __( 'Error:', 'wp-travel-pro' ) . '</strong> ' . __( 'Your Item has been added but the email could not be sent.Possible reason: your host may have disabled the mail() function.', 'wp-travel-pro' ), 'error' );
			}
		}

		public static function invoice_template( $booking_id, $template = 'default' ) {
			include sprintf( '%s/inc/invoice-templates/default.php', self::$abspath );

		}

		public static function notices() {
			// Invoice dir
			if ( ! file_exists( WP_TRAVEL_INVOICE_PATH ) ) {
				$created = mkdir( WP_TRAVEL_INVOICE_PATH, 0755, true );
				if ( ! $created ) {
					?>
					<strong> <?php _e( 'Error:', 'wp-travel-pro' ); ?></strong><?php _e( 'Unable to create directory wt-invoice.', 'wp-travel-pro' ); ?>
					<?php
				}
			}

			if ( ! is_readable( WP_TRAVEL_INVOICE_PATH ) || ! is_writable( WP_TRAVEL_INVOICE_PATH ) ) {
				?>
				<strong> <?php _e( 'Error:', 'wp-travel-pro' ); ?></strong>
								   <?php
									echo WP_TRAVEL_INVOICE_PATH;
									_e( ' is not readable/writable.', 'wp-travel-pro' );
									?>
				<?php
			}

			// Invoice tmp dir
			if ( ! file_exists( WP_TRAVEL_INVOICE_TMP_PATH ) ) {
				$created = mkdir( WP_TRAVEL_INVOICE_TMP_PATH, 0755, true );
				if ( ! $created ) {
					?>
					<strong> <?php _e( 'Error:', 'wp-travel-pro' ); ?></strong><?php _e( 'Unable to create directory tmp.', 'wp-travel-pro' ); ?>
					<?php
				}
			}

			if ( ! is_readable( WP_TRAVEL_INVOICE_TMP_PATH ) || ! is_writable( WP_TRAVEL_INVOICE_TMP_PATH ) ) {
				?>
				<strong> <?php _e( 'Error:', 'wp-travel-pro' ); ?></strong>
								   <?php
									echo WP_TRAVEL_INVOICE_TMP_PATH;
									_e( ' is not readable/writable.', 'wp-travel-pro' );
									?>
				<?php
			}
		}

		public static function display_notices( $display ) {
			// Invoice dir
			if ( ! file_exists( WP_TRAVEL_INVOICE_PATH ) ) {
				$created = mkdir( WP_TRAVEL_INVOICE_PATH, 0755, true );
				if ( ! $created ) {
					$display = true;
				}
			}

			if ( ! is_readable( WP_TRAVEL_INVOICE_PATH ) || ! is_writable( WP_TRAVEL_INVOICE_PATH ) ) {
				$display = true;
			}

			// Invoice tmp dir
			if ( ! file_exists( WP_TRAVEL_INVOICE_TMP_PATH ) ) {
				$created = mkdir( WP_TRAVEL_INVOICE_TMP_PATH, 0755, true );
				if ( ! $created ) {
					$display = true;
				}
			}

			if ( ! is_readable( WP_TRAVEL_INVOICE_TMP_PATH ) || ! is_writable( WP_TRAVEL_INVOICE_TMP_PATH ) ) {
				$display = true;
			}
			return $display;
		}

		public static function download_pdf( $booking_id ) {
			if ( ! $booking_id ) {
				return;
			}
			$invoice_id = get_post_meta( $booking_id, 'wt_invoice_id', true );

			if ( ! $invoice_id ) {
				return;
			}

			$filename = $invoice_id . '.pdf';
			$dir      = trailingslashit( WP_TRAVEL_INVOICE_PATH );

			if ( ! file_exists( $dir . $filename ) ) {
				return;
			}
			?>
			<a href="<?php echo esc_url( content_url( '/wt-invoice/' ) . $filename ); ?>" style="text-align:right;display:block"><?php _e( 'PDF', 'wp-travel-pro' ); ?></a>
			<?php
		}
	}
endif;
