<?php
/**
 * WP Travel Inventory Management Core Class.
 *
 * @package WP_Travel_Inventory_Management
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Inventory Management core.
if ( ! class_exists( 'WP_Travel_Inventory_Management_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Inventory_Management_Core {

		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Inventory Management Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Inventory_Management_Core Instance.
		 * Ensures only one instance of WP_Travel_Inventory_Management_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Inventory_Management_Core()
		 * @return WP_Travel_Inventory_Management_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];

			// if ( function_exists( 'wptravel_get_settings' ) ) {
			// 	$settings = wptravel_get_settings();
			// } else {
			// 	$settings = wp_travel_get_settings();
			// }

			// $enable_utilities = isset( $settings['show_wp_travel_inventory_magagement'] ) ? $settings['show_wp_travel_inventory_magagement'] : 'yes';

			add_filter( 'wp_travel_settings_fields', array( 'WP_Travel_Inventory_Management_Core', 'settings_fields' ) );
			add_action( 'wp_travel_addons_setings_tab_fields', array( 'WP_Travel_Inventory_Management_Core', 'plugin_action' ) );
			// if ( 'yes' !== $enable_utilities ) {
			// add_action( 'wp_travel_trip_inventory_tab_content', array( 'WP_Travel_Inventory_Management_Core', 'plugin_disabled_notice' ) );
			// return;
			// }

			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Inventory_Management_Core', 'assets' ) );
			add_action( 'init', array( 'WP_Travel_Inventory_Management_Core', 'includes' ) );

			// Inventory Option Tab Content in admin Single Trip.
			add_action( 'wp_travel_trip_inventory_tab_content', array( 'WP_Travel_Inventory_Management_Core', 'inventory_option_content' ), 10, 2 );

			// Save Inventory Option Tab in admin Single Trip.
			add_filter( 'wp_travel_save_trip_metas', array( 'WP_Travel_Inventory_Management_Core', 'inventory_option_save_content' ) );

			/**
			 * Gallery Data Filter WP Travel v4
			 *
			 * @since 1.0.3
			 */
			add_filter( 'wp_travel_trip_data', array( __CLASS__, 'add_rest_data' ), 11, 2 );
			add_action( 'wp_travel_update_trip_data', array( __CLASS__, 'save_rest_data' ), 11, 2 );

			// Inventory Ajax.
			include sprintf( '%s/core/ajax/inventory.php', self::$abspath );

			// Inventory Helpers
			include sprintf( '%s/core/helpers/inventory.php', self::$abspath );
			include sprintf( '%s/core/helpers/error_codes.php', self::$abspath );
			include sprintf( '%s/core/helpers/response_codes.php', self::$abspath );

		}

		
		/**
		 * Meta fields added by the addon.
		 *
		 * @since react // TODO: Update version on release.
		 */
		private static function get_trip_meta_fields( $trip_id ) {
			return array(
				'enable_trip_inventory'                => array(
					'default' => 'no',
					'type'    => 'string',
					'options' => array( 'yes', 'no' ),
				),
				'status_column_enable'                 => array(
					'default' => 'no',
					'type'    => 'string',
					'options' => array( 'yes', 'no' ),
				),
				'wp_travel_inventory_pax_limit_type'   => array(
					'default' => 'use_group_size',
					'type'    => 'string',
					'options' => array( 'use_group_size', 'custom_value' ),
				),
				'wp_travel_inventory_sold_out_action'  => array(
					'default' => 'show_sold_out_msg_only',
					'type'    => 'string',
					'options' => array( 'show_sold_out_msg_only', 'allow_trip_enquiry' ),
				),
				'wp_travel_inventory_custom_max_pax'   => array(
					'default' => absint( get_post_meta( $trip_id, 'wp_travel_group_size', true ) ),
					'type'    => 'integer',
				),
				'wp_travel_inventory_size'             => array(
					'default' => 1,
					'type'    => 'integer',
				),
				'wp_travel_inventory_sold_out_message' => array(
					'default' => 'Sold Out!',
					'type'    => 'string',
				),
				'wp_travel_inventory_status_message_format' => array(
					'default' => '',
					'type'    => 'string',
				),
				'inventory_counts_paid_only'           => array(
					'default' => 'no',
					'type'    => 'string',
				),
			);
		}

		/**
		 * Adds Inventory data new rest api.
		 *
		 * @since react
		 */
		public static function add_rest_data( $trip_data, $trip_id = 0 ) {
			$inventory_data = array();
			foreach ( (array) self::get_trip_meta_fields( $trip_id ) as $meta_key => $field_schema ) {
				$meta_value = get_post_meta( $trip_id, $meta_key, true );
				if ( isset( $field_schema['options'] ) && ! in_array( $meta_value, $field_schema['options'] ) ) {
					$meta_value = $field_schema['default'];
				}
				// $inventory_data[ $meta_key ] = ! empty( $meta_value ) && $field_schema['type'] === gettype( $meta_value ) ? $meta_value : $field_schema['default'];
				$inventory_data[ $meta_key ] = ! empty( $meta_value ) ? $meta_value : $field_schema['default'];
			}

			$trip_data['inventory'] = $inventory_data;

			return $trip_data;
		}

		public static function save_rest_data( $trip_data, $trip_id = 0 ) {
			$inventory_data = isset( $trip_data->inventory ) ? $trip_data->inventory : array();
			$valid_fields   = self::get_trip_meta_fields( $trip_id );

			foreach ( (array) $inventory_data as $name => $value ) {
				if ( array_key_exists( $name, $valid_fields ) ) {
					// $value_to_update = gettype( $value ) === $valid_fields[$name]['type'] ? $value : $valid_fields[$name]['default'];
					$value_to_update = $value ? $value : $valid_fields[ $name ]['default'];
				} else {
					continue;
				}

				update_post_meta( $trip_id, $name, $value_to_update );
			}
		}

		/**
		 * Default settings fields.
		 *
		 * @param array $settings WP Travel Settings.
		 * @return array
		 */
		public static function settings_fields( $settings ) {
			$settings['show_wp_travel_inventory_magagement'] = 'yes';
			return $settings;
		}

		/**
		 * Plugin action to show / hide plugin settings and features.
		 */
		public static function plugin_action() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$field_name  = 'show_wp_travel_inventory_magagement';
			$field_label = __( 'WP Travel Inventory Management', 'wp-travel-pro' );

			$field_value = isset( $settings[ $field_name ] ) ? $settings[ $field_name ] : 'yes'; ?>
			<table class="form-table">
				<tr>
					<th>
						<label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label>
					</th>
					<td>
						<span class="show-in-frontend checkbox-default-design">
							<label data-on="ON" data-off="OFF">
								<input value="no" name="<?php echo esc_attr( $field_name ); ?>" type="hidden" />
								<input <?php checked( $field_value, 'yes' ); ?> value="yes" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" type="checkbox" />
								<span class="switch"></span>
							</label>
						</span>
						<p class="description"><label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html__( 'Show all your "', 'wp-travel-pro' ) . esc_html( $field_label ) . esc_html__( '" settings and enable its feature.', 'wp-travel-pro' ); ?></label></p>
					</td>
				</tr>
			</table>
			<?php

		}

		public static function plugin_disabled_notice() {
			echo sprintf( '<p><strong>WP Travel Inventory Management</strong> is currently disabled please go to <a href="%s">Addons Settings</a> to enable it. </p>', esc_url( admin_url( 'edit.php?post_type=itinerary-booking&page=settings#wp-travel-tab-content-addons_settings' ) ) );
		}

		/**
		 * Inventory Management assets.
		 */
		public static function assets() {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

			wp_enqueue_script( 'wp-travel-inventory-admin-script', self::$plugin_url . 'assets/js/admin-script' . $suffix . '.js', array( 'jquery' ), self::$version, true );

			// React Stuffs. @since WP Travel v4.0.0
			$deps = include_once sprintf( '%sapp/build/wp-travel-inventory-management-admin-trip-options.asset.php', plugin_dir_path( __FILE__ ) );
			wp_enqueue_script( 'wp-travel-inventory-management-admin-trip-options', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-inventory-management-admin-trip-options.js', $deps['dependencies'], $deps['version'], true );
			// wp_enqueue_style( 'wp-travel-inventory-management-admin-trip-options-style', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-inventory-management-admin-trip-options.css', array( 'wp-components' ), $deps['version'] );

		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}

			/**
			 * Include required files
			 */
			include sprintf( '%s/inc/class-wp-travel-utilities-inventory.php', self::$abspath );

		}

		/**
		 * Inventory Option tab content in admin single trip.
		 *
		 * @param array $args tab arguments.
		 * @package wp-travel-inventory-management
		 */
		public static function inventory_option_content( $args ) {

			$post_id = $args['post']->ID;

			$enable_trip_inventory = get_post_meta( $post_id, 'enable_trip_inventory', true );
			$status_column_enable  = get_post_meta( $post_id, 'status_column_enable', true );

			if ( ! $enable_trip_inventory ) {

				$enable_trip_inventory = 'no';

			}

			if ( ! $status_column_enable ) {

				$status_column_enable = 'yes';

			}

			$wp_travel_inventory_pax_limit_type = get_post_meta( $post_id, 'wp_travel_inventory_pax_limit_type', true );

			if ( ! $wp_travel_inventory_pax_limit_type ) {

				$wp_travel_inventory_pax_limit_type = 'use_group_size';

			}

			$wp_travel_inventory_sold_out_action = get_post_meta( $post_id, 'wp_travel_inventory_sold_out_action', true );

			if ( ! $wp_travel_inventory_sold_out_action ) {

				$wp_travel_inventory_sold_out_action = 'allow_trip_enquiry';

			}

			$display                                   = 'use_group_size' === $wp_travel_inventory_pax_limit_type ? 'style="display:none"' : 'style="display:table-row"';
			$sold_out_style                            = 'allow_trip_enquiry' === $wp_travel_inventory_sold_out_action ? 'style="display:none"' : 'style="display:table-row"';
			$wp_travel_inventory_custom_max_pax        = get_post_meta( $post_id, 'wp_travel_inventory_custom_max_pax', true );
			$wp_travel_inventory_sold_out_message      = get_post_meta( $post_id, 'wp_travel_inventory_sold_out_message', true );
			$wp_travel_inventory_status_message_format = get_post_meta( $post_id, 'wp_travel_inventory_status_message_format', true );

			if ( ! $wp_travel_inventory_status_message_format ) {

				$wp_travel_inventory_status_message_format = __( 'Pax Available: {available_pax} / {pax_limit}', 'wp-travel-pro' );

			}

			$inventory = new WP_Travel_Util_Inventory();

			$booked_pax_no  = $inventory->get_booking_pax_count( $post_id );
			$min_custom_pax = 1;

			if ( $booked_pax_no && 0 < $booked_pax_no ) {

				$min_custom_pax = $booked_pax_no;

			}
			?>
			<table class="form-table">
			<tr>
				<th>
					<label for="enable_trip_inventory"><?php esc_html_e( 'Enable Trip Inventory', 'wp-travel-pro' ); ?></label>
				</th>
				<td>
					<span class="show-in-frontend checkbox-default-design">
						<label data-on="ON" data-off="OFF">
							<input <?php checked( $enable_trip_inventory, 'yes' ); ?> value="yes" name="enable_trip_inventory" id="enable_trip_inventory" type="checkbox" />
							<span class="switch">
							</span>
						</label>
					</span>
					<p class="description"><label for="enable_trip_inventory"><?php esc_html_e( 'Enable to activate inventory features for this trip !!', 'wp-travel-pro' ); ?></label></p>
				</td>
			</tr>
			<!--
			<tr>
				<th>
					<label for="status_column_enable"><?php esc_html_e( 'Status Column', 'wp-travel-pro' ); ?></label>
				</th>
				<td>
					<span class="show-in-frontend checkbox-default-design">
						<label data-on="ON" data-off="OFF">
							<input <?php checked( $status_column_enable, 'yes' ); ?> value="yes" name="status_column_enable" id="status_column_enable" type="checkbox" />
							<span class="switch">
							</span>
						</label>
					</span>
					<p class="description"><label for="status_column_enable"><?php esc_html_e( 'Show Inventory Status Column on Bookings Tab !!', 'wp-travel-pro' ); ?></label></p>
				</td>
			</tr>-->
			<tr>
				<th>
					<label for="wp_travel_inventory_status_message_format"><?php esc_html_e( 'Status Message Format:', 'wp-travel-pro' ); ?></label>
				</th>
				<td>
					<textarea id="wp_travel_inventory_status_message_format" name="wp_travel_inventory_status_message_format" rows="3" cols="30"><?php echo esc_html( $wp_travel_inventory_status_message_format ); ?></textarea>
					<p class="description"><?php esc_html_e( 'Status message format. Tags available {available_pax},{booked_pax},{pax_limit}', 'wp-travel-pro' ); ?></p>
				</td>
			</tr>

			<tr>
				<th>
					<label for="currency"><?php esc_html_e( 'Pax Limit', 'wp-travel-pro' ); ?></label>
				</th>
				<td>
					<label><input <?php checked( $wp_travel_inventory_pax_limit_type, 'use_group_size' ); ?> type="radio" value="use_group_size" name="wp_travel_inventory_pax_limit_type"> <?php esc_html_e( 'Use Pricing Group Size Value', 'wp-travel-pro' ); ?></label><br>
					<label><input <?php checked( $wp_travel_inventory_pax_limit_type, 'custom_value' ); ?> type="radio" value="custom_value" name="wp_travel_inventory_pax_limit_type"><?php esc_html_e( 'Custom Value', 'wp-travel-pro' ); ?></label>
				</td>
			</tr>
			<tr id="wp-travel-inventory-custom-max-pax-size-row" <?php echo esc_attr( $display ); ?> >
				<th>
					<label for="wp-travel-inventory-custom-max-pax-size"><?php esc_html_e( 'Max Pax no.', 'wp-travel-pro' ); ?></label>
				</th>
				<td>
					<input type="number" min="<?php echo esc_attr( $min_custom_pax ); ?>" step="1" value="<?php echo esc_attr( $wp_travel_inventory_custom_max_pax ); ?>" id="wp-travel-inventory-custom-max-pax-size" name="wp_travel_inventory_custom_max_pax">
				</td>
			</tr>
			<tr>
				<th>
					<label for="currency"><?php esc_html_e( '"OUT OF STOCK" Options:', 'wp-travel-pro' ); ?></label>
				</th>
				<td>
					<label><input <?php checked( $wp_travel_inventory_sold_out_action, 'allow_trip_enquiry' ); ?> type="radio" value="allow_trip_enquiry" name="wp_travel_inventory_sold_out_action"><?php esc_html_e( 'Display Enquiry Button', 'wp-travel-pro' ); ?></label><br>
					<label><input <?php checked( $wp_travel_inventory_sold_out_action, 'show_sold_out_msg_only' ); ?> type="radio" value="show_sold_out_msg_only" name="wp_travel_inventory_sold_out_action"><?php esc_html_e( 'Show "SOLD OUT" Message', 'wp-travel-pro' ); ?></label>
				</td>
			</tr>
			<tr id="wp-travel-inventory-sold-out-msg" <?php echo esc_attr( $sold_out_style ); ?> >
				<th>
					<label for="wp_travel_inventory_sold_out_message"><?php esc_html_e( 'Sold Out Message:', 'wp-travel-pro' ); ?></label>
				</th>
				<td>
					<textarea id="wp_travel_inventory_sold_out_message" name="wp_travel_inventory_sold_out_message" rows="5" cols="30"><?php echo esc_attr( $wp_travel_inventory_sold_out_message ); ?></textarea>
					<p class="description"><?php esc_html_e( 'Custom message for unavailable trips!!', 'wp-travel-pro' ); ?></p>
				</td>
			</tr>
			</table>
			<?php
		}

		/**
		 * Inventory Option save content meta.
		 *
		 * @package wp-travel-inventory-management
		 */
		public static function inventory_option_save_content( $trip_meta ) {

			$enable_trip_inventory = 'no';
			if ( isset( $_POST['enable_trip_inventory'] ) ) {
				$enable_trip_inventory = sanitize_text_field( wp_unslash( $_POST['enable_trip_inventory'] ) );
			}
			$trip_meta['enable_trip_inventory'] = $enable_trip_inventory;

			$status_column_enable = 'no';
			if ( isset( $_POST['status_column_enable'] ) ) {
				$status_column_enable = sanitize_text_field( wp_unslash( $_POST['status_column_enable'] ) );
			}
			$trip_meta['status_column_enable'] = $status_column_enable;

			if ( isset( $_POST['wp_travel_inventory_pax_limit_type'] ) ) {
				$wp_travel_inventory_pax_limit_type              = $_POST['wp_travel_inventory_pax_limit_type'];
				$trip_meta['wp_travel_inventory_pax_limit_type'] = $wp_travel_inventory_pax_limit_type;
			}

			if ( isset( $_POST['wp_travel_inventory_sold_out_action'] ) ) {
				$wp_travel_inventory_sold_out_action              = $_POST['wp_travel_inventory_sold_out_action'];
				$trip_meta['wp_travel_inventory_sold_out_action'] = $wp_travel_inventory_sold_out_action;
			}

			if ( isset( $_POST['wp_travel_inventory_custom_max_pax'] ) ) {
				$wp_travel_inventory_custom_max_pax              = $_POST['wp_travel_inventory_custom_max_pax'];
				$trip_meta['wp_travel_inventory_custom_max_pax'] = $wp_travel_inventory_custom_max_pax;
			}

			if ( isset( $_POST['wp_travel_inventory_sold_out_message'] ) ) {
				$wp_travel_inventory_sold_out_message              = $_POST['wp_travel_inventory_sold_out_message'];
				$trip_meta['wp_travel_inventory_sold_out_message'] = $wp_travel_inventory_sold_out_message;
			}

			if ( isset( $_POST['wp_travel_inventory_status_message_format'] ) ) {
				$wp_travel_inventory_status_message_format              = $_POST['wp_travel_inventory_status_message_format'];
				$trip_meta['wp_travel_inventory_status_message_format'] = $wp_travel_inventory_status_message_format;
			}
			return $trip_meta;

		}

	}
endif;
