<?php
/**
 * Inventory Helper Calss.
 *
 * @package WP_Travel_Inventory_Management
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Inventory Helper Function.
 */
class WP_Travel_Helpers_Inventory {
	/**
	 * Get Inventory.
	 *
	 * @param array $args Inventory arguments.
	 *
	 * @return array
	 */
	public static function get_inventory( $args ) {

		$trip_id       = $args['trip_id'];
		$pricing_id    = $args['pricing_id'];
		$selected_date = $args['selected_date'];

		if ( empty( $trip_id ) ) {
			return WP_Travel_Helpers_Error_Codes::get_error( 'WP_TRAVEL_NO_TRIP_ID' );
		}
		if ( empty( $pricing_id ) ) {
			return WP_Travel_Helpers_Error_Codes::get_error( 'WP_TRAVEL_NO_PRICING_ID' );
		}
		if ( empty( $selected_date ) ) {
			return WP_Travel_Helpers_Error_Codes::get_error( 'WP_TRAVEL_NO_DATE' );
		}
		if ( ! self::is_inventory_enabled( $trip_id ) ) {
			return WP_Travel_Helpers_Error_Codes::get_error( 'WP_TRAVEL_INVENTORY_NOT_ENABLED' );
		}

		$times          = $args['times'];
		$trip_times     = ! empty( $times ) ? explode( ',', $times ) : array();
		$inventory_data = array();
		if ( is_array( $trip_times ) && count( $trip_times ) > 0 ) {
			foreach ( $trip_times as $trip_time ) {

				$inventory_key = self::get_inventory_key( $pricing_id, $selected_date, $trip_time );
				$pax_limit     = self::get_pricing_option_pax_limit( $trip_id, $pricing_id ); // Group Size calculation [Custom inventory size | Group size | pricing max pax].

				$booked_pax = get_post_meta( $trip_id, $inventory_key, true );
				$booked_pax = $booked_pax ? $booked_pax : 0;

				$available_pax = $pax_limit - $booked_pax;
				if ( $available_pax < 0 ) {
					$available_pax = 0;
				}

				$data             = array(
					'date'          => sprintf( '%sT%s', $selected_date, $trip_time ),
					'pax_available' => (int) $available_pax,
					'booked_pax'    => (int) $booked_pax,
					'pax_limit'     => (int) $pax_limit,
				);
				$inventory_data[] = $data;
			}
		} else { // In case of only date.
			$inventory_key = self::get_inventory_key( $pricing_id, $selected_date );

			$pax_limit = self::get_pricing_option_pax_limit( $trip_id, $pricing_id ); // Group Size calculation [Custom inventory size | Group size | pricing max pax].

			$booked_pax = get_post_meta( $trip_id, $inventory_key, true );
			$booked_pax = $booked_pax ? $booked_pax : 0;

			$available_pax = $pax_limit - $booked_pax;
			if ( $available_pax < 0 ) {
				$available_pax = 0;
			}

			// Sample Response.
			$data             = array(
				'date'          => sprintf( '%sT%s', $selected_date, '00:00' ),
				'pax_available' => (int) $available_pax,
				'booked_pax'    => (int) $booked_pax,
				'pax_limit'     => (int) $pax_limit,
			);
			$inventory_data[] = $data;
		}

		return array(
			'code'      => 'WP_TRAVEL_INVENTORY_INFO',
			'inventory' => $inventory_data,
		);
	}

	/**
	 * Update Inventory.
	 *
	 * @param array $args Inventory arguments.
	 *
	 * @return mixed
	 */
	public static function update_inventory( $args ) {

		$settings       = wptravel_get_settings();
		$trip_id       = $args['trip_id'];
		$pricing_id    = $args['pricing_id'];
		$selected_date = $args['selected_date'];
		$time          = isset( $args['time'] ) ? $args['time'] : '';
		$pax           = absint( $args['pax'] );

		if ( empty( $trip_id ) ) {
			return WP_Travel_Helpers_Error_Codes::get_error( 'WP_TRAVEL_NO_TRIP_ID' );
		}
		if ( empty( $pricing_id ) ) {
			return WP_Travel_Helpers_Error_Codes::get_error( 'WP_TRAVEL_NO_PRICING_ID' );
		}
		if ( empty( $selected_date ) ) {
			return WP_Travel_Helpers_Error_Codes::get_error( 'WP_TRAVEL_NO_DATE' );
		}
		if ( empty( $pax ) ) {
			return WP_Travel_Helpers_Error_Codes::get_error( 'WP_TRAVEL_NO_PAX' );
		}
		if ( ! self::is_inventory_enabled( $trip_id ) ) {
			return WP_Travel_Helpers_Error_Codes::get_error( 'WP_TRAVEL_INVENTORY_NOT_ENABLED' );
		}

		$sanitized_data      = wptravel_sanitize_array( $_POST ); // @phpcs:ignore

		if( class_exists( 'WP_Travel_Pro' ) && isset( $settings['selected_booking_option'] ) && count( $settings['selected_booking_option'] ) == 1 && $settings['selected_booking_option'][0] = 'booking-with-payment' ){
			$sanitized_data['wp_travel_booking_option'] = 'booking_with_payment';
		}
		$paid_inventory_only = get_post_meta( $trip_id, 'inventory_counts_paid_only', true );
		$booking_option      = isset( $sanitized_data['wp_travel_booking_option'] ) ? $sanitized_data['wp_travel_booking_option'] : 'booking_only';
		
		if( class_exists( 'WP_Travel_Pro' ) && isset( $settings['selected_booking_option'] ) && count( $settings['selected_booking_option'] ) == 1 && $settings['selected_booking_option'][0] = 'booking-with-payment' ){
			$sanitized_data['wp_travel_booking_option'] = 'booking_with_payment';
		}

		if ( ! $paid_inventory_only ) {
			$paid_inventory_only = 'no';
		}
		if ( 'yes' === $paid_inventory_only && 'booking_only' === $booking_option ) {
			return false;
		}

		$inventory_data = array();

		$inventory_key = self::get_inventory_key( $pricing_id, $selected_date, $time );
		$booked_pax    = get_post_meta( $trip_id, $inventory_key, true );
		$booked_pax    = $booked_pax ? (int) $booked_pax : 0;

		$pax_limit = self::get_pricing_option_pax_limit( $trip_id, $pricing_id ); // Group Size calculation [Custom inventory size | Group size | pricing max pax].

		$available_pax = $pax_limit - $booked_pax;
		if ( $available_pax > 0 ) {
			if ( $pax <= $available_pax ) {
				$booked_pax += $pax;
				update_post_meta( $trip_id, $inventory_key, $booked_pax );
				return WP_Travel_Helpers_Response_Codes::get_success_response( 'WP_TRAVEL_INVENTORY_PAX_UPDATED' );
			} else {
				$args = array(
					'allowed' => $available_pax,
					'entered' => $pax,
				);
				return WP_Travel_Helpers_Error_Codes::get_error( 'WP_TRAVEL_INVENTORY_EXCEED', $args );
			}
		} else {
			return WP_Travel_Helpers_Error_Codes::get_error( 'WP_TRAVEL_INVENTORY_FULL' );
		}
		return false;

	}

	/* =======Pirivate======= */

	/**
	 * Get Booked pax key for inventory.
	 *
	 * @param Number $pricing_id Pricing ID Required.
	 * @param Date   $trip_date  Trip Date Required.
	 * @param Time   $trip_time  Trip Time Optional.
	 */
	public static function get_inventory_key( $pricing_id = false, $trip_date = false, $trip_time = false ) {

		$meta_name = 'wt_booked_pax';
		if ( $pricing_id ) {
			$meta_name .= sprintf( '-%s', $pricing_id ); // Added to work with trip pricing_id.
		}
		if ( $trip_date ) {
			$meta_name .= sprintf( '-%s', str_replace( '-', '_', $trip_date ) ); // Added to work with trip date.
		}
		if ( $trip_time ) {
			$meta_name .= sprintf( '-%s', str_replace( ':', '_', $trip_time ) ); // Added to work with trip time.
		} else {
			$meta_name .= '-00_00'; // Added to work with trip time.
		}
		return $meta_name;
	}

	/**
	 * Return Pricing Options Pax Limit.
	 */
	public static function get_pricing_option_pax_limit( $trip_id, $pricing_id ) {
		if ( empty( $trip_id ) || ! self::is_inventory_enabled( $trip_id ) ) {
			return false;
		}

		if ( empty( $pricing_id ) ) {
			return false;
		}

		$pax_limit = 0;

		$inventory_size = get_post_meta( $trip_id, 'wp_travel_inventory_size', true );
		$pricing_data   = WP_Travel_Helpers_Pricings::get_pricings( $trip_id );
		if ( isset( $pricing_data['code'] ) && 'WP_TRAVEL_TRIP_PRICINGS' === $pricing_data['code'] ) {
			$pricings = $pricing_data['pricings'];

			$index = array_search( $pricing_id, array_column( $pricings, 'id' ) );

			if ( 'boolean' !== gettype( $index ) ) {
				$pricing = $pricings[ $index ];
				$max_pax = $pricing['max_pax'];

				if ( $max_pax > 0 ) { // 1st Set Max Pax as Inventory size.
					$pax_limit = $max_pax;
				} elseif ( $inventory_size > 0 ) { // 2nd Set Group Size as  Inventory size.
					$pax_limit = $inventory_size;
				} else { // 3rd Set Custom size as  Inventory size.
					$pax_limit = $custom_inventory_size;
				}
			}
		}

		return $pax_limit;
	}

	/**
	 * Check if the inventory is enabled.
	 *
	 * @param $trip_id
	 * @return bool true | false
	 */
	public static function is_inventory_enabled( $trip_id ) {
		if ( empty( $trip_id ) ) {
			$inventory_enabled = false;
		}
		$inventory         = get_post_meta( $trip_id, 'enable_trip_inventory', true );
		$inventory_enabled = 'yes' === $inventory ? true : false;
		return apply_filters( 'wp_travel_filter_inventory_enabled', $inventory_enabled );
	}

}
