<?php
/**
 * CSV Exporter Class for WP Travel
 *
 * @package WP_Travel\inc\admin
 */
class WP_Travel_CSV_Exporter {


	/**
	 * The Post Type to be exported.
	 *
	 * @var string
	 */
	private $post_type;

	/**
	 * Supported Export Post-types by the Plugin.
	 *
	 * @var array
	 */
	private $allowed_post_types;

	/**
	 * Current Post Obeject.
	 *
	 * @var object
	 */
	private $post;

	/**
	 * Class Constructor.
	 */
	public function __construct() {
		$this->allowed_post_types = array(
			'itinerary-booking',
			WP_TRAVEL_POST_TYPE,
			'tour-extras',
			'wp-travel-payment',
			'itinerary-enquiries',
			'wp-travel-coupons',
		);
		// Initialize hooks.
		$this->init_hooks();

	}

	/**
	 * Initialize hooks.
	 */
	public function init_hooks() {
		add_action( 'export_wp', array( $this, 'init_exporter' ) );

		add_action( 'admin_notices', array( $this, 'add_export_button' ) );
	}

	/**
	 * Setups Exporter.
	 *
	 * @param array $args Arguments.
	 * @return void
	 */
	public function init_exporter( $args ) {
		$this->post_type = $args['content'];

		if ( $this->is_supported_post_type( $this->post_type ) ) {
			$this->csv_generator( $args );
		}
	}


	/**
	 * Adds Export Button to the supported post-types page.
	 */
	public function add_export_button() {
		$screen = get_current_screen();
		if ( 'export' === $screen->id ) {

			if ( isset( $_GET['error'] ) && 'empty' === $_GET['error'] ) {
				?>
				<div class="notice notice-error"><p><?php esc_html_e( 'Error: No data found for selected post type.', 'wp-travel-pro' ); ?></p></div>
				<?php
			}
		}

		$csv_supported = array( 'edit-itinerary-booking', 'edit-' . WP_TRAVEL_POST_TYPE, 'edit-tour-extras', 'edit-itinerary-enquiries', 'edit-wp-travel-coupons' );
		if ( ! in_array( $screen->id, $csv_supported, true ) ) {
			return;
		}

		?>
		<div class="wrap export-to-csv">
			<a href="<?php echo admin_url( 'export.php' ); ?>" class="page-title-action"><?php esc_html_e( 'Export to CSV', 'wp-travel-pro' ); ?></a>
			<a href="<?php echo admin_url( 'admin.php?import=wp-travel-import-export' ); ?>" class="page-title-action"><?php esc_html_e( 'Import CSV', 'wp-travel-pro' ); ?></a>
		</div>
		<?php
	}

	/**
	 * Process content of CSV file
	 **/
	public function csv_generator( $args ) {

		// Check for current user privileges.
		if ( ! current_user_can( 'manage_options' ) || ! is_admin() ) {
			return false;
		}

		$meta_keys = $this->get_meta_keys();
		array_push( $meta_keys, 'pricing_v4', 'dates_v4' ); // Added for v4 import/export support.
		$posts     = $this->get_posts();

		$this->generate_csv( $posts, $meta_keys );
		// Exports JSON
		// $this->generate_json( $posts, $meta_keys );
	}

	/**
	 * Gets all the meta keys of the posts of Specified post type.
	 *
	 * @return array
	 */
	public function get_meta_keys() {
		global $wpdb;
		$query  = '';
		$query .= " SELECT DISTINCT($wpdb->postmeta.meta_key)";
		$query .= " FROM $wpdb->posts";
		$query .= " LEFT JOIN $wpdb->postmeta";
		$query .= " ON $wpdb->posts.ID = $wpdb->postmeta.post_id";
		$query .= " WHERE $wpdb->posts.post_type = '%s'";
		$query .= " AND $wpdb->postmeta.meta_key != ''";
		$query .= " AND $wpdb->postmeta.meta_key NOT RegExp '(^[_0-9].+$)'";
		$query .= " AND $wpdb->postmeta.meta_key NOT RegExp '(^[0-9]+$)'";

		return $wpdb->get_col( $wpdb->prepare( $query, $this->post_type ) );
	}

	/**
	 * Get Posts for the Post Type.
	 *
	 * @return mixed
	 */
	private function get_posts() {
		$posts = get_posts(
			array(
				'post_type'      => $this->post_type,
				'post_status'    => array( 'publish', 'pending', 'draft', 'future', 'private', 'inherit', 'trash' ),
				'posts_per_page' => -1,
			)
		);
		if ( ! $posts ) {
			$referer = add_query_arg( 'error', 'empty', wp_get_referer() );
			wp_redirect( $referer );
			exit;
		}
		return $posts;
	}

	/**
	 * Checks if the exporting post-type is supported.
	 *
	 * @param string $post_type Post Type
	 * @return boolean
	 */
	private function is_supported_post_type( $post_type = null ) {
		$post_type = $post_type ? $post_type : $this->post_type;
		if ( $post_type ) {
			return in_array( $post_type, $this->allowed_post_types, true );
		}
		return false;
	}

	/**
	 * CSV File Name.
	 *
	 * @return string
	 */
	private function csv_name() {
		$sitename = sanitize_key( get_bloginfo( 'name' ) );
		if ( ! empty( $sitename ) ) {
			$sitename .= '.';
		}
		return $sitename . $this->post_type . date( 'Y-m-d-H-i-s' ) . '.csv';
	}

	/**
	 * CSV Header.
	 *
	 * @param array   $fields Fields.
	 * @param boolean $echo IF Echo or return header fields array.
	 * @return array
	 */
	private function csv_data_header( $fields, $echo = true ) {
		$additional   = array( 'post_title', 'post_content', 'post_excerpt', 'post_date', 'post_status', 'post_name', 'taxonomies', '_thumbnail_id', 'post_type' );
		$headers      = array_merge( $additional, $fields );
		$exclude_data = apply_filters( 'wp_travel_booking_data_exclude_data', array() );
		$headers      = array_diff( $headers, $exclude_data );
		if ( $echo ) {
			echo implode( ',', $headers ) . "\n";
		}
		return $headers;
	}

	/**
	 * Gets post field value.
	 *
	 * @param string $value Meta Value.
	 * @return void
	 */
	private function csv_field_value( $key ) {
		$post = $this->post;
		switch ( $key ) {
			case 'taxonomies':
				$taxonomies     = array();
				$taxonomies_arr = get_post_taxonomies( $post->ID );
				if ( ! empty( $taxonomies_arr ) && is_array( $taxonomies_arr ) ) {
					foreach ( $taxonomies_arr as $t_k => $tax ) {
						$terms              = get_the_terms( $post->ID, $tax );
						$taxonomies[ $tax ] = $terms;
					}
				}
				$value = $taxonomies;
				break;
			case 'wp_travel_itinerary_gallery_ids':
				$gallery_images = isset( $post->{$key} ) ? $post->{$key} : array();

				if ( is_array( $gallery_images ) && ! empty( $gallery_images ) ) {
					foreach ( $gallery_images as $key => $id ) {
						$gallery_images[ $key ] = wp_get_attachment_url( $id );
					}
				}
				$value = $gallery_images;
				break;
			case '_thumbnail_id':
				$value = wptravel_get_post_thumbnail_url( $post->ID, 'full' );
				break;
			case 'post_type':
				$value = get_post_type( $post->ID );
				break;
			case 'post_date':
				$value = get_the_date( 'Y/m/d H:i:s', $post->ID );
				break;
			case 'dates_v4':
				if ( class_exists( 'WP_Travel_Helpers_Trip_Dates' ) ) :
					$data = WP_Travel_Helpers_Trip_Dates::get_dates( $post->ID );
					if ( is_array( $data ) ) {
						$value = ! empty( $data['dates'] ) ? ( $data['dates'] ) : array();
					} else {
						$value = array();
					}
				endif;
				break;
			case 'pricing_v4':
				if ( class_exists( 'WP_Travel_Helpers_Pricings' ) ) :
					$data = WP_Travel_Helpers_Pricings::get_pricings( $post->ID );
					if ( is_array( $data ) ) {
						$value = ! empty( $data['pricings'] ) ? ( $data['pricings'] ) : array();
					} else {
						$value = array();
					}
				endif;
				break;
			default:
				$value = isset( $post->{$key} ) ? $post->{$key} : '';
				break;
		}
		$value = is_array( $value ) ? serialize( $value ) : $value;
		return $value;
	}

	/**
	 * Undocumented function
	 *
	 * @param array   $posts Posts Array.
	 * @param array   $fields Fields Array.
	 * @param boolean $echo Echo or not.
	 * @return array
	 */
	private function csv_data( $posts, $fields, $echo = false ) {

		$data = array();
		if ( is_array( $posts ) ) {
			foreach ( $posts as $post ) {
				$this->post = $post;
				$metas      = array();
				foreach ( $fields as $field ) {
					$value                       = $this->csv_field_value( $field );
					$metas[]                     = '"' . str_replace( '"', '""', $value ) . '"';
					$data[ $post->ID ][ $field ] = $value;
				}
				if ( $echo ) {
					echo implode( ',', $metas ) . "\n";
					
				}
			}
		}
		return $data;
	}

	/**
	 * Sets Headers for export file.
	 *
	 * @return void
	 */
	private function set_header() {
		header( 'Content-Description: File Transfer' );
		header( 'Content-Disposition: attachment; filename=' . $this->csv_name() );
		header( 'Content-Type: text/csv; charset=utf-8', true );
	}

	/**
	 * Generates Final Export File.
	 *
	 * @param array $posts Posts Array.
	 * @param array $meta_keys Metas
	 * @return void
	 */
	public function generate_csv( $posts, $meta_keys ) {
		$this->set_header();
		$this->csv_data_header( $meta_keys );
		$this->csv_data( $posts, $this->csv_data_header( $meta_keys, false ), true );
		exit;
	}

	/**
	 * Exports JSON.
	 *
	 * @param [type] $posts Posts Array.
	 * @param [type] $meta_keys Metas.
	 * @return void
	 */
	public function generate_json( $posts, $meta_keys ) {
		$this->set_header();
		$_data    = array();
		$defaults = array( 'post_title', 'post_content', 'post_excerpt', 'post_date', 'post_status', 'post_name', 'taxonomies', '_thumbnail_id' );
		foreach ( $posts as $post ) {
			$this->post = $post;
			foreach ( $defaults as $key ) {
				$_data[ $post->ID ][ $key ] = $post->{$key};
				if ( '_thumbnail_id' === $key ) {
					$_data[ $post->ID ]['_thumbnail_id'] = wptravel_get_post_thumbnail_url( $post->ID, 'full' );
				}
				if ( 'taxonomies' === $key ) {
					$this->csv_field_value( 'taxonomies' );
					$_data[ $post->ID ]['taxonomies'] = $this->csv_field_value( 'taxonomies' );
				}
			}
			foreach ( $meta_keys as $meta ) {
				$_data[ $post->ID ]['metas'][ $meta ] = is_array( $post->{$meta} ) ? serialize( $post->{$meta} ) : $post->{$meta};
				if ( 'wp_travel_itinerary_gallery_ids' === $meta ) {
					$_data[ $post->ID ]['metas']['wp_travel_itinerary_gallery_ids'] = $this->csv_field_value( 'wp_travel_itinerary_gallery_ids' );
				}
			}
		}
		$data[ $this->post_type ] = $_data;
		// $data['trips'] = $this->csv_data( $posts, $this->csv_data_header( $meta_keys, false ), false );
		$data = json_encode( $data );
		echo $data;
		exit;
	}

}

new WP_Travel_CSV_Exporter();
