<?php
/**
 * WP Travel Group Discount Core Class.
 *
 * @package wp-travel-tour-extras-core
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Group Discount core.
if ( ! class_exists( 'WP_Travel_Group_Discount_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Group_Discount_Core {
		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Group Discount Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Group_Discount_Core Instance.
		 * Ensures only one instance of WP_Travel_Group_Discount_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Group_Discount_Core()
		 * @return WP_Travel_Group_Discount_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$enable_group_discount = isset( $settings['show_wp_travel_group_discount'] ) ? $settings['show_wp_travel_group_discount'] : 'yes';

			add_filter( 'wp_travel_settings_fields', array( 'WP_Travel_Group_Discount_Core', 'settings_fields' ) );
			add_action( 'wp_travel_addons_setings_tab_fields', array( 'WP_Travel_Group_Discount_Core', 'plugin_action' ) );
			// if ( 'yes' !== $enable_group_discount ) {
			// return;
			// }

			self::includes();
			add_action( 'wp_enqueue_scripts', array( 'WP_Travel_Group_Discount_Core', 'frontend_assets' ) );
			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Group_Discount_Core', 'admin_assets' ) );

			// Group Discount fields on individual trips.
			// add_action( 'wp_travel_pricing_option_content_after_trip_extra', 'wp_travel_group_discount_admin_fields', 10, 3 );
			add_action( 'wp_travel_pricing_option_content_after_category', 'wp_travel_group_discount_admin_fields', 10, 4 );
			add_action( 'wp_travel_pricing_option_content_after_category_repeator', 'wp_travel_group_discount_admin_fields', 10, 4 );

			add_action( 'wp_travel_pricing_option_content_after_category_inside_repeator', 'wp_travel_group_discount_admin_fields_repeator', 10, 2 );

			// Save group discount meta.
			add_filter( 'wp_travel_save_trip_metas', 'wp_travel_group_discount_admin_fields_save', 10, 2 );

			// Display Group discount list in frontend.
			add_action( 'wp_travel_booking_after_select_button', 'wp_travel_group_discount_pricing_list', 10, 2 );
		}

		/**
		 * Default settings fields.
		 *
		 * @param array $settings WP Travel Settings.
		 * @return array
		 */
		public static function settings_fields( $settings ) {
			$settings['show_wp_travel_group_discount'] = 'yes';
			return $settings;
		}

		/**
		 * Plugin action to show / hide plugin settings and features.
		 */
		public static function plugin_action() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$field_name  = 'show_wp_travel_group_discount';
			$field_label = __( 'WP Travel Group Discount', 'wp-travel-pro' );

			$field_value = isset( $settings[ $field_name ] ) ? $settings[ $field_name ] : 'yes'; ?>
			<table class="form-table">
				<tr>
					<th>
						<label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label>
					</th>
					<td>
						<span class="show-in-frontend checkbox-default-design">
							<label data-on="ON" data-off="OFF">
								<input value="no" name="<?php echo esc_attr( $field_name ); ?>" type="hidden" />
								<input <?php checked( $field_value, 'yes' ); ?> value="yes" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" type="checkbox" />
								<span class="switch"></span>
							</label>
						</span>
						<p class="description"><label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html__( 'Show all your "', 'wp-travel-pro' ) . esc_html( $field_label ) . esc_html__( '" settings and enable its feature.', 'wp-travel-pro' ); ?></label></p>
					</td>
				</tr>
			</table>
			<?php

		}

		/**
		 * Frontend assets.
		 */
		public static function frontend_assets() {

			$suffix          = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			$switch_to_react = false;
			if ( function_exists( 'wptravel_is_react_version_enabled' ) ) {
				$switch_to_react = wptravel_is_react_version_enabled();
			} else {
				$switch_to_react = wp_travel_is_react_version_enabled();
			}
			if ( ! function_exists( 'wp_travel_is_react_version_enabled' ) || ! $switch_to_react ) :
				wp_enqueue_style( 'wp-travel-group-discount-frontend-css', self::$plugin_url . 'assets/css/wp-travel-group-discount-frontend' . $suffix . '.css', '', self::$version );
				wp_enqueue_script( 'wp-travel-group-discount-frontend-script', self::$plugin_url . 'assets/js/wp-travel-group-discount-frontend.js', array( 'jquery' ), self::$version, true );
			endif;
		}

		/**
		 * Admin assets.
		 */
		public static function admin_assets() {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			// wp_enqueue_style( 'wp-travel-group-discount-admin-css', self::$plugin_url . 'assets/css/wp-travel-group-discount-admin' . $suffix . '.css', '', self::$version );

			wp_enqueue_script( 'wp-travel-group-discount-admin-script', self::$plugin_url . 'assets/js/wp-travel-group-discount-admin.js', array( 'jquery' ), self::$version, true );

			$deps = include_once sprintf( '%sapp/build/admin-group-discount.asset.php', self::$abspath );
			wp_enqueue_script( 'wp-travel-group-discount-admin-script1', self::$plugin_url . 'app/build/admin-group-discount.js', $deps['dependencies'], $deps['version'], true );

		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}
			/**
			 * Include required files
			 */
			include sprintf( '%sinc/helpers.php', self::$abspath );
			include sprintf( '%sinc/functions.php', self::$abspath );
			if ( self::is_request( 'admin' ) ) {
				include sprintf( '%sinc/admin/admin-functions.php', self::$abspath );
			}
		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}
	}
endif;
