<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel fields editor core.
if ( ! class_exists( 'WP_Travel_Field_Editor_Core' ) ) :
	class WP_Travel_Field_Editor_Core {

		protected static $ABSPATH;

		protected static $PLUGIN_FILE;

		protected static $VERSION;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Field Editor Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Field_Editor_Core Instance.
		 * Ensures only one instance of WP_Travel_Field_Editor_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Field_Editor_Core()
		 * @return WP_Travel_Field_Editor_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Default settings fields.
		 *
		 * @param array $settings WP Travel Settings.
		 * @return array
		 */
		public static function settings_fields( $settings ) {
			$settings['show_wp_travel_field_editor'] = 'yes';
			return $settings;
		}

		/**
		 * Init core.
		 */
		public static function init( $params ) {
			self::$PLUGIN_FILE = __FILE__;
			self::$ABSPATH     = dirname( self::$PLUGIN_FILE ) . '/';
			self::$VERSION     = $params['version'];

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$enable_field_editor = isset( $settings['show_wp_travel_field_editor'] ) ? $settings['show_wp_travel_field_editor'] : 'yes';
			add_filter( 'wp_travel_settings_fields', array( 'WP_Travel_Field_Editor_Core', 'settings_fields' ) );
			add_action( 'wp_travel_addons_setings_tab_fields', array( 'WP_Travel_Field_Editor_Core', 'plugin_action' ) );
			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Field_Editor_Core', 'assets' ) );
			
			// if ( 'yes' !== $enable_field_editor ) {
			// add_action( 'wp_travel_settings_tab_field_editor_fields', array( 'WP_Travel_Field_Editor_Core', 'plugin_disabled_notice' ) );
			// return;
			// }

			self::includes();

			add_filter( 'wp_travel_register_field_types', array( 'WP_Travel_Field_Editor_Core', 'register_field_types' ) );

			add_action( 'wp_enqueue_scripts', array( 'WP_Travel_Field_Editor_Core', 'frontend_assets' ) );
			add_filter( 'script_loader_tag', array( 'WP_Travel_Field_Editor_Core', 'add_script_attribute' ), 10, 2 );
			add_filter( 'wp_travel_settings_tabs', array( 'WP_Travel_Field_Editor_Core', 'register_settings_tabs' ) );
			add_action( 'wp_travel_tabs_content_settings_field_editor', array( 'WP_Travel_Field_Editor_Core', 'settings_tab_content' ), 10, 2 );
			add_action( 'wp_travel_after_admin_setting_form', array( 'WP_Travel_Field_Editor_Core', 'add_popup_after_admin_setting_form' ) );
			add_action( 'wp_travel_before_save_settings', array( 'WP_Travel_Field_Editor_Core', 'save_settings' ) );

			add_filter( 'wp_travel_enquiries_form_fields', array( 'WP_Travel_Field_Editor_Core', 'enquiries_form_fields' ) );
			add_filter( 'wp_travel_checkout_traveller_fields', array( 'WP_Travel_Field_Editor_Core', 'traveller_form_fields' ) );
			add_filter( 'wp_travel_checkout_billing_fields', array( 'WP_Travel_Field_Editor_Core', 'checkout_billing_fields' ) );
			add_action( 'wp_ajax_wp_travel_field_editor_get_fields', array( 'WP_Travel_Field_Editor_Core', 'load_fields' ) );

			add_action( 'wp_travel_frontend_enquiry_data', array( 'WP_Travel_Field_Editor_Core', 'process_enquiry_data' ), 10 );
			add_action( 'wp_travel_admin_enquiry_data', array( 'WP_Travel_Field_Editor_Core', 'process_enquiry_data' ), 10 );

			add_filter( 'wp_travel_frontend_enqueries_validation_check', array( 'WP_Travel_Field_Editor_Core', 'validation_check' ) );

			// add_action( 'wp_travel_tabs_content_settings', array( 'WP_Travel_Field_Editor_Core', 'misc_tab_call_back' ), 25, 2 );
			add_action( 'wp_travel_tabs_content_settings_misc_options_global', array( 'WP_Travel_Field_Editor_Core', 'misc_tab_call_back' ), 15, 2 );

			add_filter( 'wp_travel_before_save_settings', array( 'WP_Travel_Field_Editor_Core', 'third_party_save_settings' ) );

			add_filter( 'wp_travel_admin_enquery_email_tags', array( 'WP_Travel_Field_Editor_Core', 'enquery_email_tags' ), 10, 2 );
			add_filter( 'wp_travel_admin_booking_email_tags', array( 'WP_Travel_Field_Editor_Core', 'booking_email_tags' ), 10 );

			add_filter( 'wp_travel_require_all_travelers_fields', array( __CLASS__, 'require_all_travelers_fields' ) );

			/**
			 * @since 1.1.3
			 */
			// add_action( 'wp_head', array( __CLASS__, 'inline_css' ) );
		}

		/**
		 * Fix for select box design issue of trip enquiry form.
		 */
		public static function inline_css() {
			echo '<style>';
			?>
			#wp-travel-enquiries select {
				height: auto;
			}
			<?php
			echo '</style>';
		}

		/**
		 * Plugin action to show / hide plugin settings and features.
		 */
		public static function plugin_action() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$field_name  = 'show_wp_travel_field_editor';
			$field_label = __( 'WP Travel Field Editor', 'wp-travel-pro' );

			$field_value = isset( $settings[ $field_name ] ) ? $settings[ $field_name ] : 'yes';
			?>
			<table class="form-table">
				<tr>
					<th>
						<label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label>
					</th>
					<td>
						<span class="show-in-frontend checkbox-default-design">
							<label data-on="ON" data-off="OFF">
								<input value="no" name="<?php echo esc_attr( $field_name ); ?>" type="hidden" />
								<input <?php checked( $field_value, 'yes' ); ?> value="yes" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" type="checkbox" />
								<span class="switch"></span>
							</label>
						</span>
						<p class="description"><label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html__( 'Show all your "', 'wp-travel-pro' ) . esc_html( $field_label ) . esc_html__( '" settings and enable its feature.', 'wp-travel-pro' ); ?></label></p>
					</td>
				</tr>
			</table>
			<?php

		}

		public static function plugin_disabled_notice() {
			echo sprintf( '<p><strong>WP Travel Field Editor</strong> is currently disabled please go to <a href="%s" class="wp-travel-field-editor-enable-link">Addons Settings</a> to enable it. </p>', esc_url( admin_url( 'edit.php?post_type=itinerary-booking&page=settings#wp-travel-tab-content-addons_settings' ) ) );
			?>
			<script>
				jQuery(document).ready( function($){
				$( document ).on( 'click', '.wp-travel-field-editor-enable-link', function(e){
					e.preventDefault();
					$( '#wp-travel-tab-addons_settings a' ).trigger( 'click' );
				} );
				} );
			</script>
			<?php
		}

		/**
		 * Include class/function fields.
		 */
		public static function includes() {
			include_once WP_TRAVEL_ABSPATH . 'inc/framework/form/class.form.php';
			include sprintf( '%s/inc/fields/class.field.recaptcha-v2.php', self::$ABSPATH );
			include sprintf( '%s/inc/translation/translate-field-labels.php', self::$ABSPATH ); // @since 1.0.9

			if ( is_admin() ) {
				include sprintf( '%s/core/helpers/settings.php', self::$ABSPATH ); // @since 1.0.9
				include sprintf( '%s/core/ajax/settings.php', self::$ABSPATH ); // @since 1.0.9
			}
		}

		public static function register_field_types( $field_types ) {
			if ( version_compare( WP_TRAVEL_VERSION, '1.9.0' ) >= 0 ) {
				$field_types['recaptcha_v2'] = array(
					'label' => __( 'Recaptcha V2', 'wp-travel-pro' ),
					'class' => 'WP_Travel_FW_Field_Recaptcha_V2',
				);
			} else {
				$field_types['recaptcha_v2'] = 'WP_Travel_FW_Field_Recaptcha_V2';
			}
			return $field_types;
		}

		/**
		 * Field editor assets.
		 */
		public static function assets() {
			if ( function_exists( 'wptravel_is_admin_page' ) ) {
				if ( ! wptravel_is_admin_page( array( 'settings' ) ) ) {
					return;
				}
			} else {
				if ( ! wp_travel_is_admin_page( array( 'settings' ) ) ) {
					return;
				}
			}
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

			$translation_array = array(
				'add_field'            => __( '+ Add field', 'wp-travel-pro' ),
				'remove'               => __( 'Remove', 'wp-travel-pro' ),
				'enable'               => __( 'Enable', 'wp-travel-pro' ),
				'disable'              => __( 'Disable', 'wp-travel-pro' ),
				'reset_field_title'    => __( 'Reset to default fields', 'wp-travel-pro' ),
				'reset_field'          => __( 'Reset', 'wp-travel-pro' ),
				'label'                => __( 'Label', 'wp-travel-pro' ),
				'name'                 => __( 'Name', 'wp-travel-pro' ),
				'type'                 => __( 'Type', 'wp-travel-pro' ),
				'enabled'              => __( 'Enabled', 'wp-travel-pro' ),
				'action'               => __( 'Action', 'wp-travel-pro' ),
				'delete_field_message' => __( 'Are you sure to delete this field?', 'wp-travel-pro' ),
				'reset_message'        => __( 'Are you sure you want to reset to default fields? all your changes will be deleted.', 'wp-travel-pro' ),
				'fields'               => self::fields_options(),
				'validation_messages'  => array(
					'validname' => __( 'Field name already exists or invalid.', 'wp-travel-pro' ),
					'validid'   => __( 'Field id already exists or invalid.', 'wp-travel-pro' ),
				),
			);

			wp_register_script( 'wp-travel-field-editor-views', plugins_url( '/assets/js/dist/field-editor-views' . $suffix . '.js', self::$PLUGIN_FILE ), array( 'jquery', 'jquery-ui-sortable' ), self::$VERSION, true );
			wp_register_script( 'wp-travel-field-editor', plugins_url( '/assets/js/dist/field-editor' . $suffix . '.js', self::$PLUGIN_FILE ), array( 'jquery', 'jquery-ui-sortable', 'wp-travel-field-editor-views' ), self::$VERSION, true );

			wp_localize_script( 'wp-travel-field-editor-views', 'wp_travel_field_editor', $translation_array );

			wp_enqueue_style( 'magnific-popup-css' );
			wp_enqueue_script( 'magnific-popup-script' );
			// wp_enqueue_script( 'wp-travel-field-editor-views' );
			wp_enqueue_script( 'wp-travel-field-editor' );

			// Styles
			wp_register_style( 'wp-travel-field-editor', plugins_url( '/assets/css/wp-travel-field-editor' . $suffix . '.css', self::$PLUGIN_FILE ), array(), self::$VERSION );
			wp_enqueue_style( 'wp-travel-field-editor' );

			// settings2 screen.
			$deps                   = include_once sprintf( '%s/app/build/wp-travel-field-editor-settings.asset.php', plugin_dir_path( __FILE__ ) );
			$deps['dependencies'][] = 'jquery-parsley';
			wp_enqueue_script( 'wp-travel-field-editor-settings', plugin_dir_url( __FILE__ ) . 'app/build/wp-travel-field-editor-settings.js', $deps['dependencies'], $deps['version'], true );
		}

		/**
		 * Frontend assets.
		 */
		public static function frontend_assets() {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

			/**
			 * Styles @since 1.0.9
			 * Moved snippet to wp_head hook @since 1.1.3
			 */
			// wp_register_style( 'wp-travel-field-editor-public', plugins_url( '/assets/css/wp-travel-field-editor-public.css', self::$PLUGIN_FILE ), array(), self::$VERSION );
			// wp_enqueue_style( 'wp-travel-field-editor-public' );

			// Scripts
			wp_register_script( 'grecaptcha', 'https://www.google.com/recaptcha/api.js', array(), self::$VERSION, false );
			wp_enqueue_script( 'grecaptcha' );
			// wp_register_script( 'wp-travel-field-editor-frontend', plugins_url( '/assets/js/dist/field-editor-frontend' . $suffix . '.js', self::$PLUGIN_FILE ), array( 'jquery', 'grecaptcha' ), self::$VERSION, true );

			// wp_enqueue_script( 'grecaptcha' );
			// wp_enqueue_script( 'wp-travel-field-editor-frontend' ); // Seems not required @since
		}

		/**
		 * Add script attrs.
		 */
		public static function add_script_attribute( $tag, $handle ) {
			if ( 'grecaptcha' !== $handle ) {
				return $tag;
			}
			return str_replace( ' src', '  async defer src', $tag );
		}

		/**
		 * Register settings tabs.
		 */
		public static function register_settings_tabs( $settings_fields ) {
			// TODO: move before misc tab.
			if ( ! isset( $settings_fields['field_editor'] ) ) {

				$settings_fields['field_editor'] = array(
					'tab_label'     => __( 'Field Editor', 'wp-travel-pro' ),
					'content_title' => __( 'Field Editor', 'wp-travel-pro' ),
					'priority'      => 75,
				);
			}
			return $settings_fields;
		}

		/**
		 * Settings tab content.
		 *
		 * @param  string $tab  Current tab.
		 * @param  array  $args Settings args.
		 */
		public static function settings_tab_content( $tab, $args ) {
			include sprintf( '%s/views/tab_content.php', self::$ABSPATH );
		}

		/**
		 * Add popup after settings form.
		 */
		public static function add_popup_after_admin_setting_form() {
			include sprintf( '%s/views/popup_content.php', self::$ABSPATH );
		}

		/**
		 * Save field editor settings.
		 *
		 * @param  array $settings Other settings.
		 * @return array           Modified settings.
		 */
		public static function save_settings( $settings ) {
			$form_fields     = array();
			$form_old_fields = get_option( 'wp_travel_forms_properties', array() );
			$locale          = wp_travel_field_editor_get_locale();

			/**
			 * Support for translation.
			 *
			 * @since 1.0.9
			 */
			if ( isset( $form_old_fields[ $locale ] ) ) {
				$form_old_fields[ $locale ] = get_option( 'wp_travel_forms_properties', array() );
				$form_old_fields            = ! empty( $form_old_fields[ $locale ] ) ? $form_old_fields[ $locale ] : array();
			}

			if ( ! empty( $_POST['field_property'] ) ) :
				$field_property = $_POST['field_property'];
				foreach ( $field_property as $form => $fields ) {
					if ( ! empty( $fields ) ) {
						foreach ( $fields as $name => $field ) {
							$form_fields[ $form ][ $name ] = (array) json_decode( stripslashes_deep( $field ) );
							if ( ! empty( $form_fields[ $form ][ $name ] ) ) {
								foreach ( $form_fields[ $form ][ $name ] as $property => $value ) {
									$form_fields[ $form ][ $name ][ $property ] = is_object( $value ) ? (array) $value : $value;
								}
							}
						}
						if ( isset( $form_old_fields[ $locale ] ) ) {
							$form_old_fields[ $locale ][ $form ] = $form_fields[ $form ];
						} else {
							$form_old_fields[ $form ] = $form_fields[ $form ];
						}
					}
				}
			endif;

			/**
			 * Support for translation.
			 *
			 * @since 1.0.9
			 */
			if ( ! isset( $form_old_fields[ $locale ] ) ) {
				$form_old_fields[ $locale ] = $form_old_fields;
			}

			update_option( 'wp_travel_forms_properties', $form_old_fields );

			$settings['field_pattern_multiple_traveler'] = ! empty( $_POST['field_pattern_multiple_traveler'] ) ? 1 : 0;
			return $settings;
		}

		/**
		 * Make enquiry fields dynamic.
		 *
		 * @param  array $enquiry_fields Old fields.
		 * @return array                 Modified fields.
		 */
		public static function enquiries_form_fields( $enquiry_fields ) {
			$forms = self::get_fields( 'enquiry' );
			if ( ! empty( $forms ) ) {
				// Hide captch in admin.
				if ( is_admin() ) {
					foreach ( $forms as $key => $enquiry ) {
						if ( 'recaptcha_v2' === $enquiry['type'] ) {
							unset( $forms[ $key ] );
						}
					}
				}
				return $forms;
			}
			return $enquiry_fields;
		}

		/**
		 * Make traveller fields dynamic.
		 *
		 * @param  array $fields Old fields.
		 * @return array         Modified fields.
		 */
		public static function traveller_form_fields( $fields ) {
			$forms = self::get_fields();

			if ( isset( $forms['traveller'] ) && ! empty( $forms['traveller'] ) ) {
				$traveler_details = $forms['traveller'];
				if ( isset( $traveler_details['gender'] ) && isset( $traveler_details['gender']['options'] ) && ! empty( $traveler_details['gender']['options'] ) ) {
					$gender_data = [];
					foreach ( $traveler_details['gender']['options'] as $key => $value ) {
						$values = strtolower( $value );
						$gender_data[$values] = $value;
					}
					
					$traveler_details['gender']['options'] = $gender_data;
				}

				$cart = WP_Travel_Helpers_Cart::get_cart();
				if( is_array( $cart ) ){

					foreach( $cart['cart']['cart_items'] as $item ){
						$trip_id = $item['trip_id'];
					}
			
					if( get_post_meta( $trip_id, 'wp_travel_trip_pickup_points', true )  ){
						$locations = array();
						$required = false;
						foreach( get_post_meta( $trip_id, 'wp_travel_trip_pickup_points', true  ) as $point ){
							$locations[$point] = $point;
						}

						if( count( $locations ) > 0 ){
							$required = true;
						}

						$traveler_details['pickup_location'] = array(
							'type'        => 'select',
							'label'       => __( 'Pickup Point', 'wp-travel-pro' ),
							'name'        => 'wp_travel_pickup_location',
							'id'          => 'wp-travel-pickup-location',
							'validations' => array(
								'required' => $required,
							),
							'options'       => $locations,
							'priority'    => 90,
						);
					}
				}

				if( isset( wptravel_get_settings()['enable_multiple_checkout'] ) &&  wptravel_get_settings()['enable_multiple_checkout'] == "yes" ){ 
					unset( $traveler_details['pickup_location'] );
				}
				return $traveler_details;
			}
			return $fields;
		}

		/**
		 * Make checkout fields dynamic.
		 *
		 * @param  array $fields Old fields.
		 * @return array         Modified fields.
		 */
		public static function checkout_billing_fields( $fields ) {
			$forms = self::get_fields();

			if ( ! empty( $forms['billing'] ) ) {
				return $forms['billing'];
			}
			return $fields;
		}


		/**
		 * Load fields.
		 */
		public static function load_fields() {
			// check_ajax_referer( 'wishlists_ajax_nonce_action', 'security' );
			$form  = empty( $_POST['form'] ) ? 'enquiry' : esc_attr( $_POST['form'] );
			$reset = ! empty( $_POST['reset'] ) && 'true' === $_POST['reset'] ? true : false;

			$fields = self::load_field( $form, $reset );

			wp_send_json_success( wptravel_sort_form_fields( $fields ) );
			die();
		}

		public static function load_field( $form = 'enquiry', $reset = false ) {
			/**
			 * Support for translation.
			 *
			 * @since 1.0.9
			 */
			$locale = wp_travel_field_editor_get_locale();
			$forms  = get_option( 'wp_travel_forms_properties' );

			if ( isset( $forms[ $locale ] ) ) {
				$forms = $forms[ $locale ];
			}

			$fields         = array();
			$default_fields = array();
			if ( ! empty( $forms[ $form ] ) ) {
				$fields = $forms[ $form ];
			}

			if ( 'enquiry' === $form ) {
				// Default enquiry fields.
				$default_fields = WP_Travel_Default_Form_Fields::enquiry();
			}

			if ( 'billing' === $form ) {
				$default_fields = WP_Travel_Default_Form_Fields::billing();
			}

			if ( 'traveller' === $form ) {
				$default_fields = WP_Travel_Default_Form_Fields::traveller();
			}

			if ( $reset || empty( $fields ) ) {
				$fields = self::mark_defaults( $default_fields );
			}
			return $fields;
		}

		/**
		 * Mark fields default.
		 *
		 * @return array
		 */
		public static function mark_defaults( $fields ) {
			if ( ! empty( $fields ) ) {
				foreach ( $fields as $key => $field ) {
					$fields[ $key ]['_default'] = true;
				}
			}
			return $fields;
		}

		/**
		 * Process field values and sanitize.
		 *
		 * @param  string $form     Form to process.
		 * @param  array  $old_data  Old data.
		 * @param  array  $new_data  New Data.
		 * @return array
		 */
		public static function process_field_values( $form, $old_data, $new_data ) {
			$fields      = self::get_fields( $form );
			$field_names = array_values( wp_list_pluck( $fields, 'name' ) );
			$data        = array();
			foreach ( $field_names as $key => $name ) {
				/**
				 * Quick fix for the enquiry checkbox issue
				 *
				 * @since 1.0.8
				 */
				if ( isset( $new_data[ $name ] ) && ! empty( $new_data[ $name ] ) ) {
					$data[ $name ] = ( ! empty( $new_data[ $name ] ) && ! is_array( $new_data[ $name ] ) ) ? sanitize_text_field( $new_data[ $name ] ) : $new_data[ $name ];
				}
			}

			return wp_parse_args( $data, $old_data );
		}

		/**
		 * Process enquiry data.
		 *
		 * @param  array $enquiry_data Enquiry data
		 * @return array
		 */
		public static function process_enquiry_data( $enquiry_data ) {
			return self::process_field_values( 'enquiry', $enquiry_data, $_POST );
		}

		/**
		 * Get field editor fields.
		 *
		 * @param  string $form Form id to get fields.
		 * @return array
		 */
		public static function get_fields( $form = '' ) {
			$fields = get_option( 'wp_travel_forms_properties' );
			/**
			 * Support for translation.
			 *
			 * @since 1.0.9
			 */
			$locale = wp_travel_field_editor_get_locale();
			if ( isset( $fields[ $locale ] ) ) {
				$fields[ $locale ] = get_option( 'wp_travel_forms_properties' );
				$fields            = ! empty( $fields[ $locale ] ) ? $fields[ $locale ] : '';
			}

			if ( '' !== $form ) {
				if ( isset( $fields[ $locale ] ) ) {
					if ( isset( $fields[ $locale ][ $form ] ) ) {
						return ! empty( $fields[ $locale ][ $form ] ) ? $fields[ $locale ][ $form ] : array();
					} else {
						return ! empty( $fields[ $locale ] ) ? $fields[ $locale ] : array();
					}
				} else {
					return ! empty( $fields[ $form ] ) ? $fields[ $form ] : array();
				}
			}
			/**
			 * Support for translation.
			 *
			 * @since 1.0.9
			 */
			if ( isset( $fields[ $locale ] ) ) {
				return $fields[ $locale ];
			} else {
				return $fields;
			}
		}

		/**
		 * Fields user can choose.
		 *
		 * @return array
		 */
		public static function fields_options() {
			$fields = array();
			if ( class_exists( 'WP_Travel_FW_Field' ) ) {
				$wp_travel_fw_field = new WP_Travel_FW_Field();
				if ( ! method_exists( 'WP_Travel_FW_Field', 'register_field_types' ) ) {
					return;
				}
				$fields = $wp_travel_fw_field->register_field_types();
				$fields = wp_list_pluck( $fields, 'label' );
				// Remove unwanted fields.
				unset( $fields['category_dropdown'], $fields['text_info'], $fields['range'], $fields['date_range'], $fields['hidden'], $fields['file'] );
			}
			return $fields;
		}

		/**
		 * Trip Type option fields.
		 *
		 * @return Array
		 */
		public static function wp_travel_trip_types() {
			$fields = array();
			$terms  = get_terms(
				array(
					'taxonomy'   => 'itinerary_types',
					'hide_empty' => false,
				)
			);
			if ( is_array( $terms ) && count( $terms ) > 0 ) {
				foreach ( $terms as $key => $term ) {
					$term_id                     = ! empty( $term->term_id ) ? ( $term->term_id ) : '';
					$fields[ $term_id ]['label'] = ! empty( $term->name ) ? ( $term->name ) : '';
				}
			}
			$fields = wp_list_pluck( $fields, 'label' );
			return $fields;
		}

		/**
		 * Trip Type option fields.
		 *
		 * @return Array
		 */
		public static function wp_travel_trip_activities() {
			$fields = array();
			$terms  = get_terms(
				array(
					'taxonomy'   => 'activity',
					'hide_empty' => false,
				)
			);
			if ( is_array( $terms ) && count( $terms ) > 0 ) {
				foreach ( $terms as $key => $term ) {
					$term_id                     = ! empty( $term->term_id ) ? ( $term->term_id ) : '';
					$fields[ $term_id ]['label'] = ! empty( $term->name ) ? ( $term->name ) : '';
				}
			}
			$fields = wp_list_pluck( $fields, 'label' );
			return $fields;
		}


		/**
		 * Heading tags.
		 *
		 * @return array
		 */
		public static function heading_tags() {
			$headings = array(
				'h1'  => 'H1',
				'h2'  => 'H2',
				'h3'  => 'H3',
				'h4'  => 'H4',
				'h5'  => 'H5',
				'h6'  => 'H6',
				'p'   => 'P',
				'em'  => 'EM',
				'div' => 'DIV',
			);
			return $headings;
		}

		/**
		 * Field editor popup fields.
		 *
		 * @return array Fields.
		 */
		public static function popup_fields() {
			$fields_array = array(
				'editor_label'               => array(
					'type'        => 'text',
					'label'       => __( 'Label', 'wp-travel-pro' ),
					'name'        => 'editor_label',
					'id'          => 'editor_label',
					'validations' => array(
						'required' => true,
					),
					'attributes'  => array(
						'placeholder' => __( 'Enter field label', 'wp-travel-pro' ),
					),
					'priority'    => 10,
				),
				'editor_type'                => array(
					'type'        => 'select',
					'label'       => __( 'Type', 'wp-travel-pro' ),
					'name'        => 'editor_type',
					'id'          => 'editor_type',
					'class'       => 'wp-travel-select2',
					'validations' => array(
						'required' => true,
					),
					'attributes'  => array(
						'data-togglefields' => htmlentities(
							json_encode(
								array(
									'editor_select_options_wrap' => array(
										'show' => array( 'select', 'radio', 'checkbox' ),
									),
									'editor_heading_tags_wrap' => array(
										'show' => array( 'heading' ),
									),
									'editor_placeholder_wrap' => array(
										'hide' => array( 'heading', 'radio', 'checkbox' ),
									),
									'validation_field' => array(
										'hide' => array( 'heading', 'recaptcha_v2' ),
									),
									'editor_date_position_wrap' => array(
										'show' => array( 'date' ),
									),
								)
							),
							ENT_QUOTES,
							'UTF-8'
						),
					),
					'options'     => self::fields_options(),
					'priority'    => 15,
				),
				'editor_name'                => array(
					'type'        => 'text',
					'label'       => __( 'Name', 'wp-travel-pro' ),
					'name'        => 'editor_name',
					'id'          => 'editor_name',
					'validations' => array(
						'required' => true,
					),
					'attributes'  => array(
						'placeholder'            => __( 'Enter field name', 'wp-travel-pro' ),
						'data-parsley-validname' => 'true',
					),
					'priority'    => 20,
				),
				'editor_id'                  => array(
					'type'        => 'text',
					'label'       => __( 'ID', 'wp-travel-pro' ),
					'name'        => 'editor_id',
					'id'          => 'editor_id',
					'validations' => array(
						'required' => true,
					),
					'attributes'  => array(
						'placeholder'          => __( 'Enter field ID', 'wp-travel-pro' ),
						'data-parsley-validid' => 'true',
					),
					'priority'    => 30,
				),
				'editor_class'               => array(
					'type'       => 'text',
					'label'      => __( 'Class', 'wp-travel-pro' ),
					'name'       => 'editor_class',
					'id'         => 'editor_class',
					'attributes' => array(
						'placeholder' => __( 'Enter field class', 'wp-travel-pro' ),
					),
					'priority'   => 40,
				),
				'editor_placeholder'         => array(
					'type'          => 'text',
					'label'         => __( 'Placeholder', 'wp-travel-pro' ),
					'name'          => 'editor_placeholder',
					'id'            => 'editor_placeholder',
					'wrapper_class' => 'editor_placeholder_wrap',
					'attributes'    => array(
						'placeholder'   => __( 'Enter field placeholder', 'wp-travel-pro' ),
						'data-ishidden' => 'false',
					),
					'priority'      => 60,
				),
				'editor_select_options'      => array(
					'type'          => 'textarea',
					'label'         => __( 'Select Options', 'wp-travel-pro' ),
					'name'          => 'editor_select_options',
					'id'            => 'editor_select_options',
					'attributes'    => array(
						'placeholder'   => __( 'Enter field select options in each new line', 'wp-travel-pro' ),
						'data-ishidden' => 'true',
					),
					'wrapper_class' => 'editor_select_options_wrap',
					'priority'      => 70,
				),
				'editor_heading_tag'         => array(
					'type'          => 'select',
					'label'         => __( 'Heading Tag', 'wp-travel-pro' ),
					'name'          => 'editor_heading_tag',
					'id'            => 'editor_heading_tag',
					'class'         => 'wp-travel-select2',
					'wrapper_class' => 'editor_heading_tags_wrap',
					'attributes'    => array(
						'data-ishidden' => 'true',
					),
					'validations'   => array(
						'required' => true,
					),
					'options'       => self::heading_tags(),
					'priority'      => 80,
				),
				'editor_validation'          => array(
					'type'          => 'checkbox',
					'label'         => __( 'Validation', 'wp-travel-pro' ),
					'name'          => 'editor_validation',
					'id'            => 'editor_validation',
					'wrapper_class' => 'validation_field validation_required',
					'options'       => array(
						'required' => __( 'Required', 'wp-travel-pro' ),
					),
					'priority'      => 90,
				),
				'editor_validation_required' => array(
					'type'          => 'checkbox',
					'label'         => __( 'Required for all travelers', 'wp-travel-pro' ),
					'name'          => 'editor_validation_required',
					'id'            => 'editor_validation_required',
					'wrapper_class' => 'validation_field required_for_all',
					'options'       => array(
						'required_for_all' => __( 'Required for all travelers', 'wp-travel-pro' ),
					),
					'priority'      => 90,
				),
				'editor_date_position'       => array(
					'type'              => 'radio',
					'label'             => __( 'Date select option', 'wp-travel-pro' ),
					'name'              => 'editor_date_position',
					'id'                => 'editor_date_position',
					'wrapper_class'     => 'editor_date_position_wrap',
					'options'           => array(
						'past'   => __( 'Past dates (Including current date)', 'wp-travel-pro' ),
						'future' => __( 'Future dates (Including current date)', 'wp-travel-pro' ),
						'any'    => __( 'Any dates', 'wp-travel-pro' ),
					),
					'option_attributes' => array(
						'data-ishidden' => 'true',
					),
					'priority'          => 90,
				),
				'form_name'                  => array(
					'type' => 'hidden',
					'name' => 'form_name',
					'id'   => 'form_name',
				),
				'field_name'                => array(
					'type' => 'hidden',
					'name' => 'field_name',
					'id'   => 'field_name',
				),
			);

			return $fields_array;
		}

		public static function validation_check( $data ) {
			if ( isset( $_POST['g-recaptcha-response'] ) ) {
				$captcha = $_POST['g-recaptcha-response'];
				if ( ! $captcha ) {
					$data['status']  = false;
					$data['message'] = __( 'Please check the the captcha form.', 'wp-travel-pro' );
				}
				if ( function_exists( 'wptravel_get_settings' ) ) {
					$settings = wptravel_get_settings();
				} else {
					$settings = wp_travel_get_settings();
				}
				$secret_key = ! empty( $settings['recaptcha_v2_secret_key'] ) ? esc_attr( $settings['recaptcha_v2_secret_key'] ) : '';
				if ( empty( $secret_key ) ) {
					$data['status'] = true;
					return $data;
				}
				$ip           = $_SERVER['REMOTE_ADDR'];
				$url          = sprintf( 'https://www.google.com/recaptcha/api/siteverify?secret=%s&response=%s&remoteip=%s', $secret_key, $captcha, $ip );
				$response     = file_get_contents( $url );
				$responseKeys = json_decode( $response, true );
				if ( intval( $responseKeys['success'] ) !== 1 ) {
					$data['status']  = false;
					$data['message'] = __( 'Invalid captcha.', 'wp-travel-pro' );
				}
			}
			return $data;
		}

		public static function misc_tab_call_back( $tab, $args ) {
			if ( 'misc_options_global' !== $tab ) {
				return;
			}
			$recaptcha_v2_site_key   = ! empty( $args['settings']['recaptcha_v2_site_key'] ) ? esc_attr( $args['settings']['recaptcha_v2_site_key'] ) : '';
			$recaptcha_v2_secret_key = ! empty( $args['settings']['recaptcha_v2_secret_key'] ) ? esc_attr( $args['settings']['recaptcha_v2_secret_key'] ) : '';
			?>
			<h3 class="wp-travel-tab-content-title"><?php esc_html_e( 'reCAPTCHA (V2 Checkbox)', 'wp-travel-pro' ); ?></h3>
			<table class="form-table">
				<tr>
					<th><label for="recaptcha_v2_site_key"><?php esc_html_e( 'Site Key', 'wp-travel-pro' ); ?></label></th>
					<td>
						<input type="text" value="<?php echo esc_attr( $recaptcha_v2_site_key ); ?>" name="recaptcha_v2_site_key" id="recaptcha_v2_site_key"/>
						<p class="description"><?php esc_html_e( 'Enter reCAPTCHA site key. You can get key ', 'wp-travel-pro' ); ?><a href="https://www.google.com/recaptcha/" target="_blank"><?php esc_html_e( 'here.', 'wp-travel-pro' ); ?></a></p>
					</td>
				</tr>
				<tr>
					<th><label for="recaptcha_v2_secret_key"><?php esc_html_e( 'Secret Key', 'wp-travel-pro' ); ?></label></th>
					<td>
						<input type="text" value="<?php echo esc_attr( $recaptcha_v2_secret_key ); ?>" name="recaptcha_v2_secret_key" id="recaptcha_v2_secret_key"/>
						<p class="description"><?php esc_html_e( 'Enter reCAPTCHA Secret key. You can get key ', 'wp-travel-pro' ); ?><a href="https://www.google.com/recaptcha/" target="_blank"><?php esc_html_e( 'here.', 'wp-travel-pro' ); ?></a></p>
					</td>
				</tr>
			</table>
			<?php
		}

		/**
		 * Save settings.
		 */
		public static function third_party_save_settings( $settings ) {
			$settings['recaptcha_v2_site_key']   = ! empty( $_POST['recaptcha_v2_site_key'] ) ? esc_attr( $_POST['recaptcha_v2_site_key'] ) : '';
			$settings['recaptcha_v2_secret_key'] = ! empty( $_POST['recaptcha_v2_secret_key'] ) ? esc_attr( $_POST['recaptcha_v2_secret_key'] ) : '';

			return $settings;
		}

		/**
		 * Enquiry email tags.
		 *
		 * @param  array $email_tags Email tags.
		 * @param  array $formdata Form data.
		 * @return array           Email tags.
		 */
		public static function enquery_email_tags( $email_tags, $formdata ) {
			$forms = self::get_fields();
			if ( ! empty( $forms['enquiry'] ) ) {
				$new_email_tags = self::generate_email_tags( $forms['enquiry'], $formdata );
				$email_tags     = array_merge( $email_tags, $new_email_tags );
			}
			return $email_tags;
		}

		/**
		 * Booking email tags.
		 *
		 * @param  array $email_tags Email tags.
		 * @return array           Email tags.
		 */
		public static function booking_email_tags( $email_tags ) {
			$formdata = $_POST;
			$forms    = self::get_fields();
			if ( ! empty( $forms['billing'] ) ) {
				$new_email_tags = self::generate_email_tags( $forms['billing'], $formdata );
				$email_tags     = array_merge( $email_tags, $new_email_tags );
			}
			if ( ! empty( $forms['traveller'] ) ) {
				$new_email_tags = self::generate_email_tags( $forms['traveller'], $formdata );
				$email_tags     = array_merge( $email_tags, $new_email_tags );
			}
			return $email_tags;
		}

		/**
		 * Generate email tags based on field name.
		 *
		 * // TODO: Move this to WP Travel Core.
		 *
		 * @param  array $fields   Fields.
		 * @param  array $formdata Form data.
		 * @return array           Email tags.
		 */
		public static function generate_email_tags( $fields, $formdata ) {
			$email_tags = array();
			if ( ! empty( $fields ) ) {
				foreach ( $fields as $key => $enquiry ) {
					if ( ! isset( $enquiry['type'] ) ) {
						continue;
					}
					if ( 'recaptcha_v2' === $enquiry['type'] ) {
						// Do nothing here.
					} elseif ( 'country_dropdown' === $enquiry['type'] ) {
						$countries = wptravel_get_countries();
						// Quick Fixes for multiple traveller cases where output is in array and php warning show of illegal offset @since 4.4.4.
						if ( ! is_array( $formdata[ $enquiry['name'] ] ) ) {
							$data = isset( $countries[ $formdata[ $enquiry['name'] ] ] ) && ! empty( $countries[ $formdata[ $enquiry['name'] ] ] ) ? $countries[ $formdata[ $enquiry['name'] ] ] : '';
							if ( empty($data) ) {
								$data = $formdata[ $enquiry['name'] ];
							}
						} else if ( ! empty( $formdata[ $enquiry['name'] ] ) ) {
							$array_value = array_values( $formdata[ $enquiry['name'] ] );
							if ( is_array($array_value) && isset( $array_value[0] ) && isset( $array_value[0][0] ) ) {
								$data =  $array_value[0][0];
							} else if ( is_array($array_value) && isset( $array_value[0] ) ) {
								$data = isset($countries[[$array_value[0]]]) ? $countries[[$array_value[0]]] : '';
							} else {
								$data = isset($countries[[$array_value]]) ? $countries[[$array_value]] : '';
							}
						}
						$email_tags[ '{' . $enquiry['name'] . '}' ] = $data;
					} elseif ( 'checkbox' === $enquiry['type'] ) {
						$data = isset( $formdata[ $enquiry['name'] ] ) && ! empty( $formdata[ $enquiry['name'] ][key($formdata[ $enquiry['name'] ] )] ) && is_array($formdata[ $enquiry['name'] ][key($formdata[ $enquiry['name'] ] )]) ? implode( ', ', $formdata[ $enquiry['name'] ][key($formdata[ $enquiry['name'] ] )] ) : '';
						if ( $data === 'Array' ) {
							$data2 = isset( $formdata[ $enquiry['name'] ] ) && ! empty( $formdata[ $enquiry['name'] ] ) ? $formdata[ $enquiry['name'] ] : [];
							if ( ! empty( $data2 ) ) {
								$array_value = array_values( $data2 );
								if ( is_array($array_value) && isset( $array_value[0] ) && isset( $array_value[0][0] ) ) {
									$data = is_array( $array_value[0][0] ) ? implode( ', ', $array_value[0][0] ) : $array_value[0][0];
								} else if ( is_array($array_value) && isset( $array_value[0] ) ) {
									$data = is_array($array_value[0]) ? implode( ', ', $array_value[0] ) : $array_value[0];
								} else {
									$data = is_array($array_value[0]) ? implode( ', ', $array_value ) : $array_value;
								}
							}
						}
						$email_tags[ '{' . $enquiry['name'] . '}' ] = $data;
					} elseif ( 'radio' === $enquiry['type'] ) {
						$data = isset( $formdata[ $enquiry['name'] ] ) && ! empty( $formdata[ $enquiry['name'] ] ) && is_array($formdata[ $enquiry['name'] ]) ? $formdata[ $enquiry['name'] ] : '';
						if ( $data === 'Array' || gettype( $data ) === 'array') {
							$data2 = isset( $formdata[ $enquiry['name'] ] ) && ! empty( $formdata[ $enquiry['name'] ] ) ? $formdata[ $enquiry['name'] ] : [];
							if ( ! empty( $data2 ) ) {
								$array_value = array_values( $data2 );
								if ( is_array($array_value) && isset( $array_value[0] ) && isset( $array_value[0][0] ) ) {
									$data =  $array_value[0][0];
								} else if ( is_array($array_value) && isset( $array_value[0] ) ) {
									$data = $array_value[0];
								} else {
									$data = $array_value;
								}
							}
						}
						if( is_string($data) ) {
							$data = isset( $enquiry['options'][$data]) ? $enquiry['options'][$data] : $data;
						}
						$email_tags[ '{' . $enquiry['name'] . '}' ] = $data;
					} else {
						if ( isset( $formdata[ $enquiry['name'] ] ) && is_array( $formdata[ $enquiry['name'] ] ) ) { // Temp fixes for traveler info fields.
							foreach ( $formdata[ $enquiry['name'] ] as $value ) {
								$data = array();
								$indexs = 1;
								if ( is_array( $value ) && ! empty( $value ) ) {
									foreach ( $value as $key => $datas ) {
										$data = array_merge( $data,  array( 'Traveler', $indexs, $enquiry['label'], $datas, '<br>' ) );
										$indexs = $indexs + 1;
									}								
								}
								$email_tags[ '{' . $enquiry['name'] . '}' ] = isset( $value ) && ! empty( $value ) && ! empty( $data ) ? implode( '  ',$data ) : '';
								break;
							}
						} else {
							$email_tags[ '{' . $enquiry['name'] . '}' ] = isset( $formdata[ $enquiry['name'] ] ) && ! empty( $formdata[ $enquiry['name'] ] ) ? esc_attr( $formdata[ $enquiry['name'] ] ) : '';
						}
					}
				}
			}
			
			return $email_tags;
		}

		/**
		 * Check if all travelers fields are required or not.
		 *
		 * @param boolean $status Status of field.
		 *
		 * @return boolean
		 */
		public static function require_all_travelers_fields( $status ) {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$is_required = ! empty( $settings['field_pattern_multiple_traveler'] ) ? $settings['field_pattern_multiple_traveler'] : 0;
			return ( $is_required ) ? $is_required : $status;
		}
	}
endif;
