<?php
/**
 * This file has the required class for adding the feature for dynamic email downloadables attachments.
 *
 * ===============================================
 *			IMPORTANT IMPORTANT IMPORTANT
 * ===============================================
 * Dynamic email downloadable attachments means a feature to select the downloads from the trip backend
 * from "Downloads" tab, the selected downloads will be automatically attached to the email
 * with the help of email tag which will be dynamically rendered into html according to the
 * trip during the booking or checkout.
 *
 * This feature does not relates with the earlier added feature, which is in "downloads-email-tag-generator.php" file.
 *
 * @since 1.0.8
 * @package wp-travel-downloads
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_Travel_Downloads_Email_Downloadable_Attachments' ) ) {

	/**
	 * Class that does the heavy lifting for including the downloadables in email dynamically according the trips.
	 *
	 * @since 1.0.8
	 */
	class WP_Travel_Downloads_Email_Downloadable_Attachments {

		/**
		 * Initialize class.
		 *
		 * @return void
		 */
		public static function init() {
			add_action( 'admin_footer', array( __CLASS__, 'localized_downloads_data' ), 20 );
			add_action( 'wp_travel_trip_downloads_tab_content', array( __CLASS__, 'content_callback' ), 11, 2 );
			add_action( 'save_post', array( __CLASS__, 'save_selected_downloadables' ) );

			add_filter( 'wp_travel_admin_booking_email_tags', array( __CLASS__, 'render_html_tag' ), 12, 2 );
		}

		/**
		 * Localizes the data for wp travel downloads.
		 *
		 * @return void
		 */
		public static function localized_downloads_data() {
			$wp_travel_downloads_localized = array();

			$args = array(
				'post_type'      => 'wp_travel_downloads',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
				'orderby'        => 'title',
				'order'          => 'ASC',
			);
			$query = new WP_Query( $args );
			if ( $query->have_posts() ) {
				while ( $query->have_posts() ) {
					$query->the_post();
					$downloads_id    = get_the_ID();
					$downloads_title = get_the_title();

					$wp_travel_downloads_localized['options'][] = array(
						'value' => intval( $downloads_id ),
						'label' => $downloads_title,
					);
				}
				wp_reset_postdata();
			}

			$wp_travel_downloads_localized['i18n'] = array(
				'downloadablePlaceholder' => __( 'Click here to select downloadable attachments', 'wp-travel-pro' ),
			);

			global $current_screen;
			$current_post_type = ! empty( $current_screen->post_type ) ? $current_screen->post_type : '';
			if ( 'itineraries' === $current_post_type ) {
				$trip_id                = ! empty( $_GET['post'] ) ? $_GET['post'] : 0;
				$wp_travel_downloads    = get_post_meta( $trip_id, 'wp_travel_downloads', true );
				$selected_downloadables = ! empty( $wp_travel_downloads['email_downloadable_attachments'] ) ? $wp_travel_downloads['email_downloadable_attachments'] : array();
				if ( is_array( $selected_downloadables ) && count( $selected_downloadables ) > 0 ) {
					foreach ( $selected_downloadables as $selected_downloadable_id ) {
						$selected_downloadable_title                        = get_the_title( $selected_downloadable_id );
						$wp_travel_downloads_localized['selectedOptions'][] = array(
							'value' => intval( $selected_downloadable_id ),
							'label' => $selected_downloadable_title,
						);
					}
				}
			}

			wp_localize_script( 'wp-travel-downloads-email-downloadable-attachments', 'wp_travel_downloads_localized', $wp_travel_downloads_localized );

		}

		/**
		 * Save the wp travel downloads post data according to trip id.
		 *
		 * @param [type] $trip_id
		 * @return void
		 */
		public static function save_selected_downloadables( $trip_id ) {
			if ( isset( $_POST['wp_travel_downloads'] ) ) {
				update_post_meta( $trip_id, 'wp_travel_downloads', $_POST['wp_travel_downloads'] );
			}
		}

		/**
		 * {wp_travel_downloads_email_attachments}
		 *
		 * @return array $email_tags Booking email tag.
		 */
		public static function render_html_tag( $email_tags, $booking_id ) {
			$downloadable_link      = '';
			
			
			foreach( get_post_meta( $booking_id, 'order_items_data', true ) as $data ){

				$trip_id                = $data['trip_id'];
				$wp_travel_downloads    = get_post_meta( $trip_id, 'wp_travel_downloads', true );
				$email_tags['{wp_travel_downloads_email_attachments}'] = "";
				$selected_downloadables = ! empty( $wp_travel_downloads['email_downloads'] ) ? $wp_travel_downloads['email_downloads'] : array();
				if ( ! $selected_downloadables ) {
					$selected_downloadables = ! empty( $wp_travel_downloads['email_downloadable_attachments'] ) ? $wp_travel_downloads['email_downloadable_attachments'] : array();
				}
				$downloadable_link  .= "<h3>".esc_html( get_the_title( $trip_id )  )." ".esc_html__( ' - Attachments', 'wp-travel-pro' )."</h3>";
				if ( is_array( $selected_downloadables ) && count( $selected_downloadables ) > 0 ) {
					foreach ( $selected_downloadables as $selected_downloadable_id ) {
						$downloadables_title = get_the_title( $selected_downloadable_id );
						$download_datas      = get_post_meta( $selected_downloadable_id, 'wpt_downloads_cpt_meta', true );
						$media_id            = ! empty( $download_datas['media_id'] ) ? $download_datas['media_id'] : '';
						$downloads_url       = ! empty( $media_id ) ? wp_get_attachment_url( $media_id ) : '';
						$downloadable_link  .= "<a href='{$downloads_url}'>{$downloadables_title}</a><br>\n";
						$email_tags['{wp_travel_downloads_email_attachments}'] = $downloadable_link;
					}
				}
			}

			return $email_tags;
		}

		/**
		 * Content callback for the html in the Download tab.
		 * It is used by Reactjs later.
		 *
		 * @return void
		 */
		public static function content_callback() {
			?>
				<div id="wp-travel-downloads-email-downloadable-attachments">
					<div id="wp-travel-downloads-email-downloadable-attachments-inner"></div>
				</div>
			<?php
		}
	}
	WP_Travel_Downloads_Email_Downloadable_Attachments::init();
}
