<?php
/**
 * The file that defines the class which adds the filters functionality.
 *
 * @link       wensolutions.com
 * @since      1.0.0
 *
 * @package    WP_Travel_Custom_Filters
 * @subpackage WP_Travel_Custom_Filters/includes
 */

/**
 * The another plugin class.
 *
 * This class registers new taxonomies and add them as Trip Filters.
 *
 * @since      1.0.0
 * @package    WP_Travel_Custom_Filters
 * @subpackage WP_Travel_Custom_Filters/includes
 * @author     WEN Solutions <wensolution@gmail.com>
 */

class WP_Travel_Filters {

	/**
	 * Custom Filters option name on Options table.
	 *
	 * @var String
	 */
	public $option_name;

	/**
	 * Custom filters option Value.
	 *
	 * @var Array
	 */
	public $option_value;

	/**
	 * Core Absolute Path
	 *
	 * @var string
	 */
	private $abspath_core;

	/**
	 * Class Constructor.
	 */
	public function __construct( $abspath ) {

		$this->abspath_core = $abspath;
		$this->option_name  = 'wp_travel_custom_filters_option';
		$this->option_value = get_option( $this->option_name, array() );
		add_filter( 'wp_travel_submenus', array( $this, 'add_filter_menu' ) );

		add_action( 'admin_enqueue_scripts', array( $this, 'update_option' ) );
		add_action( 'init', array( $this, 'add_taxonomies' ) );
		add_filter( 'wp_travel_search_filter_widget_form_fields', array( $this, 'add_search_filters_fields' ) );
		add_filter( 'wp_travel_shortcode_atts', array( $this, 'add_filters_to_shortcode' ) );

		// To support is_admin_page & is_admin_pages for tax crated from custom filters.
		add_filter( 'wptravel_is_admin_pages', array( $this, 'is_admin_pages' ) );
		add_filter( 'wptravel_is_admin_page', array( $this, 'is_admin_page' ), 10, 3 );
	}

	/**
	 * Adds custom filters to WP Travel Search Filter widget.
	 *
	 * @param Array $fields Default Filters fields array.
	 * @return Array
	 */
	public function add_search_filters_fields( $fields ) {
		$taxs = $this->get_option();

		if ( is_array( $taxs ) && count( $taxs ) > 0 ) :
			$filter_widget_option = get_option( 'widget_wp_travel_filter_search_widget', array() );
			$settings = wptravel_get_settings();
			$wpml     = isset( $settings['wpml_migrations'] ) ? $settings['wpml_migrations'] : '';
			foreach ( $taxs as $slug => $value ) :
				if ( taxonomy_exists( $slug ) ) :
					$label           = isset( $value['label'] ) ? $value['label'] : '';
					if ( $wpml == true ) {
						if ( defined( 'ICL_LANGUAGE_CODE' ) ) {
							$translate_term = get_terms(
								array(
									'taxonomy'   => 'wp_travel_custom_filters',
									'hide_empty' => false,
								)
							);
							if ( ! empty( $translate_term ) ) { 
								foreach ( $translate_term as $keys => $values ) {
									$translate_slug = $values->slug ? $values->slug : '';
									$main_slug      = $slug . '-' . ICL_LANGUAGE_CODE;
									if ( ! empty( $translate_slug ) && $translate_slug == $main_slug ) {
										$label = $values->name;
									} else {
										if ( strpos( $translate_slug, $slug ) !== false ) {
											$label = $values->name;
										}
									}
								}
							}
						} 
					}
					$hierarchical    = isset( $value['hierarchical'] ) ? $value['hierarchical'] : true;
					$default         = ! empty( $_GET[ $slug ] ) ? $_GET[ $slug ] : '';
					$fields[ $slug ] = array(
						'type'            => 'category_dropdown',
						'taxonomy'        => $slug,
						'show_option_all' => esc_html__( 'All', 'wp-travel-pro' ),
						'label'           => $label,
						'name'            => $slug,
						'id'              => $slug,
						'class'           => 'wp_travel_search_widget_filters_input',
						'validations'     => array(
							'required' => false,
						),
						'default'         => $default,
						'priority'        => 45,
					);
					if ( count( $filter_widget_option ) > 0 ) {
						foreach ( $filter_widget_option as $key => $widget ) {
							if ( is_array( $widget ) && ! isset( $widget[ $slug ] ) ) {
								$widget[ $slug ]              = 1;
								$filter_widget_option[ $key ] = $widget;
							}
						}
					}
				endif; // if taxonomy_exists.
			endforeach;
			update_option( 'widget_wp_travel_filter_search_widget', $filter_widget_option );
		endif;

		return $fields;
	}
	/**
	 * Adds custom filters to WP Travel Search Filter widget.
	 *
	 * @param Array $fields Default Filters fields array.
	 * @return Array
	 */

	public function add_filters_to_shortcode( $fields ) {
		$taxs = $this->get_option();

		if ( is_array( $taxs ) && count( $taxs ) > 0 ) :
			foreach ( $taxs as $slug => $value ) :
				if ( taxonomy_exists( $slug ) ) :
					$fields[ $slug ] = 1;
				endif; // if taxonomy_exists.
			endforeach;
		endif;

		return $fields;
	}

	/**
	 * Adds Filter Menu Under Trips.
	 *
	 * @return void
	 */
	public function add_filter_menu( $submenu ) {
		$submenu['bookings']['custom_filters']['callback'] = array( $this, 'filter_page' );
		return $submenu;
	}

	/**
	 * Get Filter Page.
	 *
	 * @return void
	 */
	public function filter_page() {
		if ( ! defined( 'WP_TRAVEL_POST_TYPE' ) ) {
			return;
		}
		include_once $this->abspath_core . 'views/filter-page.php';
	}

	/**
	 * Gets Filter Table.
	 *
	 * @return void
	 */
	public function get_filters_listing_table() {
		?>
		<table class="wp-list-table widefat fixed striped">
			<thead>
				<tr>
					<td><?php esc_html_e( 'Filter Name', 'wp-travel-pro' ); ?></td>
					<td><?php esc_html_e( 'Filter Slug', 'wp-travel-pro' ); ?></td>
					<td><?php esc_html_e( 'Hierarchical', 'wp-travel-pro' ); ?></td>
					<td><?php esc_html_e( 'Show Column', 'wp-travel-pro' ); ?></td>
					<td><?php esc_html_e( 'Action', 'wp-travel-pro' ); ?></td>
				</tr>
			</thead>
			<tbody>
			<?php
			$custom_taxs = $this->get_option();
			if ( is_array( $custom_taxs ) && count( $custom_taxs ) > 0 ) {
				foreach ( $custom_taxs as $slug => $taxonomy ) {
					$tax_link          = admin_url( 'edit-tags.php?taxonomy=' . $slug . '&post_type=' . WP_TRAVEL_POST_TYPE );
					$label             = '<a href="' . esc_url( $tax_link ) . '">' . $taxonomy['label'] . '</a>';
					$tax_slug          = '<a href="' . esc_url( $tax_link ) . '">' . $slug . '</a>';
					$hierarchical      = empty( $taxonomy['hierarchical'] ) ? esc_html__( 'No', 'wp-travel-pro' ) : esc_html__( 'Yes', 'wp-travel-pro' );
					$show_admin_column = empty( $taxonomy['show_admin_column'] ) ? esc_html__( 'No', 'wp-travel-pro' ) : esc_html__( 'Yes', 'wp-travel-pro' );
					$action            = '<a href="' . add_query_arg(
						array(
							'slug'    => $slug,
							'action'  => 'remove',
							'term_id' => isset( $taxonomy['term_id'] ) && $taxonomy['term_id'] > 0 ? $taxonomy['term_id'] : 0,
						)
					) . '" class="wp_travel_trip_filter-add-filter-action" title="' . esc_html__( 'Remove', 'wp-travel-pro' ) . '" ';
					$action           .= 'data-confirm="' . esc_html__( 'The filter you selected will be permanently removed from the system.', 'wp-travel-pro' ) . '" ';
					$action           .= '>';
					$action           .= 'Remove';
					$action           .= '</a>';

					$tr  = '<tr>';
					$tr .= "<td>{$label}</td>";
					$tr .= "<td>{$tax_slug}</td>";
					$tr .= "<td>{$hierarchical}</td>";
					$tr .= "<td>{$show_admin_column}</td>";
					$tr .= "<td>{$action}</td>";
					$tr .= '</tr>';
					echo wp_kses_post( $tr );
				}
			} else {
				echo '<tr><td colspan="5">' . esc_html__( 'New Filters will be listed here.', 'wp-travel-pro' ) . '</td></tr>';
			}
			?>
			<tbody>
			<tfoot>
				<tr>
					<th><?php esc_html_e( 'Filter Name', 'wp-travel-pro' ); ?></th>
					<th><?php esc_html_e( 'Filter Slug', 'wp-travel-pro' ); ?></th>
					<th><?php esc_html_e( 'Hierarchical', 'wp-travel-pro' ); ?></th>
					<th><?php esc_html_e( 'Show Column', 'wp-travel-pro' ); ?></th>
					<th><?php esc_html_e( 'Action', 'wp-travel-pro' ); ?></td>
				</tr>
			</tfoot>
		</table>
		<?php
	}

	/**
	 * Updates the Filter Option.
	 *
	 * @return void
	 */
	public function update_option() {
		$option = $this->get_option();
		if ( ! empty( $_GET['label'] ) ) {
			$slug          = ! empty( $_GET['slug'] ) ? $_GET['slug'] : sanitize_title( $_GET['label'] );
			$label         = trim( $_GET['label'] );
			$hierarchical  = ! empty( $_GET['hierarchical'] );
			$show_in_admin = ! empty( $_GET['show_admin_column'] );
			$settings      = array();
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			}
			$wpml     = isset( $settings['wpml_migrations'] ) ? $settings['wpml_migrations'] : '';
			$term_res = '';
			if ( $wpml == true ) {
				$term_array = array(
					'slug'        => $slug,
					'description' => '',
				);
				$term_res   = wp_insert_term( $label, 'wp_travel_custom_filters', $term_array );
				if ( ! is_wp_error( $term_res ) && ! empty( $term_res ) ) {
					$trm_id = isset( $term_res['term_id'] ) ? $term_res['term_id'] : 0;
					if ( $trm_id > 0 ) {
						$term_data = get_term( $trm_id, 'wp_travel_custom_filters' );
						if ( ! is_wp_error( $term_data ) ) {
							$option[ $term_data->slug ] = array(
								'label'             => $term_data->name,
								'hierarchical'      => $hierarchical,
								'show_admin_column' => $show_in_admin,
								'term_id'           => $term_data->term_id,
							);
							$success                    = update_option( $this->option_name, $option );
							wp_redirect( remove_query_arg( array( 'slug', 'label', 'hierarchical' ) ) );
						}
					}
				}
			} else {
					$option[ $slug ] = array(
						'label'             => $label,
						'hierarchical'      => $hierarchical,
						'show_admin_column' => $show_in_admin,
					);
					$success         = update_option( $this->option_name, $option );
					wp_redirect( remove_query_arg( array( 'slug', 'label', 'hierarchical' ) ) );
			}
		} elseif ( ! empty( $_GET['slug'] ) && ! empty( $_GET['action'] ) ) {
			if ( 'remove' === $_GET['action'] ) {
				$term_id = isset( $_GET['term_id'] ) ? $_GET['term_id'] : 0;
				$term_id = (int) $term_id;
				if ( $term_id > 0 ) {
					global $wpdb;
					$db_prifix    = $wpdb->prefix;
					$trm_table    = $db_prifix . 'terms';
					$taxonomy_tbl = $db_prifix . 'term_taxonomy';
					$trm_meta_tbl = $db_prifix . 'termmeta';
					$responce     = $wpdb->delete(
						$trm_table,
						array(
							'term_id' => $term_id,
						),
						array(
							'%d',
						)
					);
					$responces    = $wpdb->delete(
						$taxonomy_tbl,
						array(
							'term_id'          => $term_id,
							'term_taxonomy_id' => $term_id,
							'taxonomy'         => 'wp_travel_custom_filters',
						),
						array(
							'%d',
							'%d',
							'%s',
						)
					);
				}
				$slug = trim( $_GET['slug'] );
				unset( $option[ $slug ] );

				update_option( 'wp_travel_custom_filters_option', $option );
				wp_safe_redirect( remove_query_arg( array( 'slug', 'action' ) ) );
			}
		}
	}
	/**
	 * Gets option Value.
	 *
	 * @return mixed
	 */
	public function get_option() {
		return $this->option_value;
	}

	/**
	 * Adds taxonomies.
	 *
	 * @return void
	 */
	public function add_taxonomies() {
		if ( ! defined( 'WP_TRAVEL_POST_TYPE' ) ) {
			return;
		}
		$taxs_to_be_registered = $this->get_option();
		$object_type           = WP_TRAVEL_POST_TYPE;
		if ( is_array( $taxs_to_be_registered ) ) {
			foreach ( $taxs_to_be_registered as $slug => $tax ) {
				$this->register_taxonomy( $slug, $tax, $object_type );
			}
		}
	}

	/**
	 * Registers new taxonomies.
	 *
	 * @param [string]       $slug New Taxonomy slug.
	 * @param [array]        $tax Arguments for new Taxonomy.
	 * @param [string/array] $object_type Post Type object.
	 * @return void
	 */
	public function register_taxonomy( $slug, $tax, $object_type ) {
		$settings = wptravel_get_settings();
		$label     = $tax['label'];
		$wpml     = isset( $settings['wpml_migrations'] ) ? $settings['wpml_migrations'] : '';
		if ( $wpml == true ) {
			if ( defined( 'ICL_LANGUAGE_CODE' ) ) {
				$translate_term = get_terms(
					array(
						'taxonomy'   => 'wp_travel_custom_filters',
						'hide_empty' => false,
					)
				);
				if ( ! empty( $translate_term ) ) { 
					foreach ( $translate_term as $keys => $values ) {
						$translate_slug = $values->slug ? $values->slug : '';
						$main_slug      = $slug . '-' . ICL_LANGUAGE_CODE;
						if ( ! empty( $translate_slug ) && $translate_slug == $main_slug ) {
							$label = $values->name;
						}
					}
				}
			} 
		}
		$args = array(
			'hierarchical'      => $tax['hierarchical'],
			'label'             => $label,
			'show_ui'           => true,
			'show_admin_column' => ! empty( $tax['show_admin_column'] ) ? $tax['show_admin_column'] : true,
			'query_var'         => true,
		);
		apply_filters( 'wp_travel_custom_filters_taxonomy_arg_filter', $args, $slug );
		register_taxonomy( $slug, $object_type, $args );
		register_taxonomy_for_object_type( $slug, $object_type );
		flush_rewrite_rules( false );
	}

	/**
	 * Add custom filters tax as pages in the page list.
	 */
	public function is_admin_pages( $pages ) {
		$taxs      = $this->get_option();
		$tax_pages = array_keys( $taxs );
		if ( is_array( $tax_pages ) && count( $tax_pages ) > 0 ) {
			$pages = array_unique( array_merge( $pages, $tax_pages ) );
		}
		return $pages;
	}

	/**
	 * Check if current page is custom taxonomy filters page or not.
	 */
	public function is_admin_page( $is_page, $slug, $screen_id ) {
		$taxs      = $this->get_option();
		$tax_pages = array_keys( $taxs );
		if ( is_array( $tax_pages ) && in_array( $slug, $tax_pages, true ) && 'edit-' . $slug === $screen_id ) {
			return true;
		}
		return $is_page;
	}

}



