<?php
/**
 * Modules core file.
 *
 * @package wp-travel-advanced-gallery-core
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Advanced Gallery Core.
if ( ! class_exists( 'WP_Travel_Advanced_Gallery_Core' ) ) :
	/**
	 * WP Travel Advanced Gallery Core.
	 */
	class WP_Travel_Advanced_Gallery_Core {

		/**
		 * Absolute path to the core module.
		 *
		 * @var string
		 */
		protected static $abspath;

		/**
		 * Plugin File.
		 *
		 * @var string
		 */
		protected static $plugin_file;

		/**
		 * The Addon Version.
		 *
		 * @var string
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP_Travel_Advanced_Gallery_Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Advanced_Gallery_Core Instance.
		 * Ensures only one instance of WP_Travel_Advanced_Gallery_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Advanced_Gallery_Core()
		 * @return WP_Travel_Advanced_Gallery_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $params Core class init paramerters.
		 */
		public static function init( array $params ) {
			self::$abspath     = dirname( __FILE__ );
			self::$plugin_file = __FILE__;
			self::$version     = $params['version'];

			add_filter( 'wp_travel_settings_fields', array( __CLASS__, 'settings_fields' ) );
			add_action( 'wp_travel_addons_setings_tab_fields', array( __CLASS__, 'plugin_action' ) );
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'assets' ) );

			add_filter( 'wp_travel_block_before_save_settings', array( __CLASS__, 'save_block_settings' ), 10, 2 );
			self::includes();
			add_action( 'init', array( __CLASS__, 'init_hooks' ) );
			/**
			 * Add Meta Fields for adv gallery.
			 */
			// add_filter( 'wptravel_itinerary_meta_fields', array( __CLASS__, 'add_meta_fields' ) );
		}

		/**
		 * React settings save.
		 *
		 * @param Array $settings WP Travel Settings.
		 * @param Array $settings_data Settings data.
		 * @return Array
		 */
		public static function save_block_settings( $settings, $settings_data ) {

			if ( isset( $settings_data['wp_travel_advanced_gallery_settings'] ) ) {
				$settings['wp_travel_advanced_gallery_settings'] = $settings_data['wp_travel_advanced_gallery_settings'];
			}

			return $settings;
		}

		/**
		 * Setting Fields.
		 *
		 * @param array $settings Settings.
		 * @return array
		 */
		public static function settings_fields( $settings ) {
			$settings['show_wp_travel_advanced_gallery'] = 'yes';
			return $settings;
		}

		/**
		 * Shows disabled notice if plugin is disabled.
		 *
		 * @return void
		 */
		public static function plugin_disabled_notice() {
			echo sprintf( '<tr><td colspan="2"><p><strong>WP Travel Advanced Gallery</strong> is currently disabled please go to <a href="%s">Addons Settings</a> to enable it. </p><td></tr>', esc_url( admin_url( 'edit.php?post_type=itinerary-booking&page=settings#wp-travel-tab-content-addons_settings' ) ) );
		}

		/**
		 * Includes Essential Files.
		 *
		 * @return void
		 */
		public static function includes() {
			include_once plugin_dir_path( __FILE__ ) . 'inc/class-wp-travel-advanced-gallery-settings.php';
			// include_once plugin_dir_path( __FILE__ ) . 'views/admin/edit-gallery-tab-content.php';
		}

		/**
		 * Adds Switch to Addon Settings Tab.
		 */
		public static function plugin_action() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$field_name  = 'show_wp_travel_advanced_gallery';
			$field_label = __( 'WP Travel Advanced Gallery', 'wp-travel-pro' );

			$field_value = isset( $settings[ $field_name ] ) ? $settings[ $field_name ] : 'yes'; ?>
			<table class="form-table">
				<tr>
					<th>
						<label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label>
					</th>
					<td>
						<span class="show-in-frontend checkbox-default-design">
							<label data-on="ON" data-off="OFF">
								<input value="no" name="<?php echo esc_attr( $field_name ); ?>" type="hidden" />
								<input <?php checked( $field_value, 'yes' ); ?> value="yes" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" type="checkbox" />
								<span class="switch"></span>
							</label>
						</span>
						<p class="description"><label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html__( 'Show all your "', 'wp-travel-pro' ) . esc_html( $field_label ) . esc_html__( '" settings and enable its feature.', 'wp-travel-pro' ); ?></label></p>
					</td>
				</tr>
			</table>
			<?php

		}

		/**
		 * Initialize Hooks.
		 */
		public static function init_hooks() {
			// Admin Hooks.
			add_filter( 'wp_travel_tabs', array( __CLASS__, 'edit_gallery_tab_callback' ), 10 ); // Changes gallery tab callback.
			add_filter( 'wp_travel_save_trip_metas', array( __CLASS__, 'add_meta_key_values' ) ); // Updates trip meta array.

			add_filter( 'wp_travel_localize_gallery_data', array( __CLASS__, 'localize_gallery_data' ) );
			// add_action( 'admin_footer', array( __CLASS__, 'script_templates' ) );

			// Public Hooks.
			add_action( 'wp_enqueue_scripts', array( __CLASS__, 'assets_views' ) );
			add_filter( 'wptravel_frontend_data', array( __CLASS__, 'localize_gallery_data_view' ), 11, 2 );
			add_filter( 'wp_travel_itinerary_tabs', array( __CLASS__, 'trip_gallery_tab' ) );

			/**
			 * Gallery Data Filter WP Travel v4
			 *
			 * @since 1.0.3
			 */
			add_filter( 'wp_travel_trip_data', array( __CLASS__, 'add_rest_gallery_data' ), 11, 2 );
			add_action( 'wp_travel_update_trip_data', array( __CLASS__, 'save_rest_gallery_data' ), 11, 2 );

			/**
			 * Filter
			 *
			 * @since 4.0.6
			 */
			add_filter(
				'wp-travel-script-handles',
				function( $handles ) {
					$handles[] = 'wp-travel-advanced-gallery-view';
					return $handles;
				}
			);
		}

		/**
		 * Adds Gallery Data for WP Travel REST.
		 *
		 * @since 1.0.3
		 */
		public static function add_rest_gallery_data( $trip_data, $trip_id ) {

			$gallery_ids      = get_post_meta( $trip_id, 'wp_travel_itinerary_gallery_ids', true );
			$advanced_gallery = get_post_meta( $trip_id, 'wp_travel_advanced_gallery', true );
			// Get Gallery data from default gallery if adv gallery have no values.
			if ( is_array( $gallery_ids ) && ( '' === $advanced_gallery || ( is_array( $advanced_gallery ) && isset( $advanced_gallery['items'] ) && empty( $advanced_gallery['items'] ) ) ) ) {
				$advanced_gallery = array();
				$index            = 0;
				foreach ( $gallery_ids as $id ) {
					if ( ! wp_get_attachment_url( $id ) ) {
						continue;
					}
					$attachment = get_post( $id );
					$media_meta = array(
						'index'        => $index,
						'attachmentid' => (int) $id,
						'id'           => (int) $id,
						'url'          => wp_get_attachment_url( $id ),
						'thumb_url'    => wp_get_attachment_thumb_url( $id ),
						'type'         => wp_check_filetype( wp_get_attachment_url( $id ) )['type'],
						'title'        => $attachment->post_title,
						'caption'      => $attachment->post_excerpt,
					);

					$advanced_gallery['items'][] = $media_meta;
					$index++;
				}
			}
			$items = ! empty( $advanced_gallery['items'] ) ? array_values( $advanced_gallery['items'] ) : array();
			if ( '' === $advanced_gallery ) { // if still empty set as empty array for advanced gallery.
				$advanced_gallery = array();
			}
			$advanced_gallery['items']     = $items;
			$trip_data['advanced_gallery'] = ! empty( $advanced_gallery ) ? $advanced_gallery : array();
			$trip_data['_thumbnail_id']    = get_post_meta( $trip_id, '_thumbnail_id', true );

			return $trip_data;

			// $gallery_data = get_post_meta( $trip_id, 'wp_travel_advanced_gallery', true );
			// $items        = ! empty( $gallery_data['items'] ) ? array_values( $gallery_data['items'] ) : array();
			// $gallery_data['items']         = $items;
			// $trip_data['advanced_gallery'] = ! empty( $gallery_data ) ? $gallery_data : array();
			// return $trip_data;
		}

		/**
		 * Update Gallery data onto trip <meta class="">
		 *
		 * @since 1.0.3
		 */
		public static function save_rest_gallery_data( $trip_data, $trip_id = 0 ) {
			$advanced_gallery_meta = array();
			if ( isset( $trip_data->{'advanced_gallery'} ) ) {
				$advanced_gallery_meta = (array) $trip_data->{'advanced_gallery'};
				$advanced_gallery_meta = array_map(
					function( $item ) {
						return (array) $item;
					},
					$advanced_gallery_meta
				);

				// Set thumbnail.
				$items = isset( $advanced_gallery_meta['items'] ) ? $advanced_gallery_meta['items'] : array();
				if ( count( $items ) > 0 ) {
					$data            = array_column( $items, 'id' );
					$video_thumb_ids = array_column( $items, 'thumb_id' );
					if ( is_array( $video_thumb_ids ) && count( $video_thumb_ids ) > 0 ) {
						$data = array_merge( $data, $video_thumb_ids );
					}

					if ( ! empty( $trip_data->_thumbnail_id ) ) {
						$_thumbnail_id = ( in_array( (int) $trip_data->_thumbnail_id, $data ) || wp_attachment_is( 'image', $trip_data->_thumbnail_id ) ) ? (int) $trip_data->_thumbnail_id : 0;
						foreach ( $data as $datas ) {
							if ( $datas == $_thumbnail_id ) {
								update_post_meta( $trip_id, '_thumbnail_id', $_thumbnail_id );
								break;
							} else {
								update_post_meta( $trip_id, '_thumbnail_id', isset( $data[0] ) ? $data[0] : 0 );
							}
						}
					} else {
						update_post_meta( $trip_id, '_thumbnail_id', isset( $data[0] ) ? $data[0] : 0 );
					}
				}
			}

			if ( count( $advanced_gallery_meta ) > 0 ) {
				update_post_meta( $trip_id, 'wp_travel_advanced_gallery', $advanced_gallery_meta );
			}

		}

		/**
		 * Callback to add Adv gallery meta fields.
		 */
		public static function add_meta_fields( $fields ) {
			// $fields['wp_travel_advanced_gallery'] = array(
			// 'show_in_rest'  => true,
			// 'single'        => true,
			// 'type'          => 'string',
			// 'auth_callback' => function () {
			// return current_user_can( 'edit_posts' );
			// },
			// );
			unset( $fields['wp_travel_itinerary_gallery_ids'] );
			return $fields;
		}

		/**
		 * Edit Gallery Tab Callback.
		 *
		 * @param string $tab Current Tab.
		 */
		public static function edit_gallery_tab_callback( $tab ) {
			$tab[ WP_TRAVEL_POST_TYPE ]['images_gallery']['callback'] = 'wp_travel_advanced_gallery_edit_gallery_tab_callback';
			return $tab;
		}

		/**
		 * Trip Gallery Content.
		 *
		 * @param array $tabs Tabs Array.
		 */
		public static function trip_gallery_tab( $tabs ) {
			$tabs['gallery']['content'] = self::trip_gallery_tab_content();
			return $tabs;
		}

		/**
		 * Templating Templates.
		 */
		// public static function script_templates() {
			// require_once plugin_dir_path( __FILE__ ) . 'views/admin/script-templates.php';
		// }

		/**
		 * Trip Gallery tab Content.
		 *
		 * @return string
		 */
		public static function trip_gallery_tab_content() {
			wp_enqueue_style( 'wp-travel-advanced-gallery-view' );
			wp_enqueue_script( 'wp-travel-advanced-gallery-view' );
			global $post;
			$gallery     = get_post_meta( $post->ID, 'wp_travel_advanced_gallery', true );
			$gallery_ids = get_post_meta( $post->ID, 'wp_travel_itinerary_gallery_ids', true ); // Legacy version.
			if ( is_array( $gallery_ids ) && ( '' === $gallery || ( is_array( $gallery ) && isset( $gallery['items'] ) && empty( $gallery['items'] ) ) ) ) {
				$gallery = array();
				$index   = 0;
				foreach ( $gallery_ids as $id ) {
					if ( ! wp_get_attachment_url( $id ) ) {
						continue;
					}
					$attachment = get_post( $id );
					$media_meta = array(
						'index'        => $index,
						'attachmentid' => (int) $id,
						'id'           => (int) $id,
						'url'          => wp_get_attachment_url( $id ),
						'thumb_url'    => wp_get_attachment_thumb_url( $id ),
						'type'         => wp_check_filetype( wp_get_attachment_url( $id ) )['type'],
						'title'        => $attachment->post_title,
						'caption'      => $attachment->post_excerpt,
					);

					$gallery['items'][] = $media_meta;
					$index++;
				}
			}
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$gallery_layout = isset( $settings['wp_travel_advanced_gallery_settings'] ) && isset( $settings['wp_travel_advanced_gallery_settings']['layout'] ) ? $settings['wp_travel_advanced_gallery_settings']['layout'] : 'grid';
			$display_type   = 'wp-travel-advanced-gallery-items-list-' . $gallery_layout;
			$content        = '<div class="wp-travel-advanced-gallery-items-list ' . $display_type . '">';
			$image_size     = 'wp_travel_thumbnail';
			if ( 'masonry' === $gallery_layout ) {
				$image_size = 'medium_large';
			}
			$size   = apply_filters( 'wp_travel_advanced_gallery_thumbnail_size', $image_size ); // 365X215
			$width  = get_option( "{$size}_size_w" );
			$height = get_option( "{$size}_size_h" );
			$size = 'medium_large';
			if ( is_array( $gallery ) && isset( $gallery['items'] ) ) :
				foreach ( $gallery['items'] as $item ) {
					
					$item       = (array) $item;
			
					$url        = wp_get_attachment_image_src( (int) $item['id'], $size ) !== false ? wp_get_attachment_image_src( (int) $item['id'], $size )[0] : $item['thumb_url'];
					$src        = '' === $url ? plugin_dir_url( __FILE__ ) . 'assets/img/wp-travel-placeholder.png' : $url;
					$class_name = ( strpos( $item['type'], 'image' ) !== false ) ? 'mfp-image' : 'mfp-iframe';
					$image_alt  = get_post_meta( $item['id'], '_wp_attachment_image_alt', true );
					$item['title'] = get_post((int) $item['id'])->post_title;
					$item['description'] = get_post((int) $item['id'])->post_content;
					$item['caption'] = get_post((int) $item['id'])->post_excerpt;
					$wrapper_class = 'image';
					if ( 'external' === $item['type'] ) {
						$wrapper_class = 'external';
					}
					if ( strpos( $item['type'], 'audio' ) !== false ) {
						$wrapper_class = 'audio';
					}
					if ( strpos( $item['type'], 'video' ) !== false ) {
						$wrapper_class = 'video';
					}
					$content .= "<div class=\"item {$wrapper_class}\"><a href=\"" . esc_url( $item['url'] ) . '" class="' . $class_name . '" title="' . $item['title'] . '" data-caption="' . $item['caption'] . '" data-description="' . $item['description'] . '">';
					$content .= '<span class="wptag__media-card">';
					$content .= '<span class="wptag__thumbnail">';
					// if ( 'external' === $item['type'] ) :
					// $content .= '<img src="' . esc_url( $src ) . '" height="' . $height . '" width="' . $width . '"/>';
					// endif;
					if ( strpos( $item['type'], 'image' ) !== false ) :
						$img_data = wp_get_attachment_image_src( (int) $item['id'], $size );
						if ( $img_data && is_array( $img_data ) && count( $img_data ) > 0 ) {
							$content .= '<img alt="' . esc_attr( $image_alt ) . '" src="' . esc_url( wp_get_attachment_image_src( (int) $item['id'], $size )[0] ) . '" title="' . esc_attr( $item['title'] ) . '"/>';
						} else {
							$content .= '<img alt="' . esc_attr( $image_alt ) . '" src="' . esc_url( $src ) . '" height="' . $height . '" width="' . $width . '"/>';
						}
					endif;
					if ( ( strpos( $item['type'], 'audio' ) !== false ) || ( strpos( $item['type'], 'video' ) !== false ) ) :
						$content .= '<img alt="' . esc_attr( $image_alt ) . '" src="' . esc_url( $src ) . '" height="' . $height . '" width="' . $width . '"/>';
					endif;
					$content .= '</span>';
					$content .= '</span>';
					$content .= '</a></div>';
				}
				else :
					$content .= sprintf( '<p>%s</p>', __( 'There are no medias on the gallery.', 'wp-travel-pro' ) );
			endif;
				$content .= '</div>';

				return $content;
		}

		/**
		 * Enqueue/Register Assets.
		 *
		 * @return void
		 */
		public static function assets() {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

			$screen         = get_current_screen();
			$allowed_screen = array( WP_TRAVEL_POST_TYPE, 'edit-' . WP_TRAVEL_POST_TYPE, 'itineraries' );

			$switch_to_react = false;
			if ( function_exists( 'wptravel_is_react_version_enabled' ) ) {
				$switch_to_react = wptravel_is_react_version_enabled();
			} else {
				$switch_to_react = wp_travel_is_react_version_enabled();
			}

			if ( function_exists( 'wptravel_is_admin_page' ) ) {
				$settings_page = wptravel_is_admin_page( array( 'settings' ) );
			} else {
				$settings_page = wp_travel_is_admin_page( array( 'settings' ) );
			}
			if ( $settings_page ) {
				$deps = include_once sprintf( '%sapp/build/wp-travel-advanced-gallery-settings.asset.php', plugin_dir_path( __FILE__ ) );
				wp_enqueue_script( 'wp-travel-advanced-gallery-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-advanced-gallery-settings.js', $deps['dependencies'], $deps['version'], true );
			}

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			
			
			if ( in_array( $screen->id, $allowed_screen ) && $switch_to_react ) {
				$deps = include_once sprintf( '%sapp/build/wp-travel-advanced-gallery-admin-trip-options.asset.php', plugin_dir_path( __FILE__ ) );
				wp_enqueue_script( 'wp-travel-advanced-gallery-admin-trip-options', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-advanced-gallery-admin-trip-options.js', $deps['dependencies'], $deps['version'], true );
				wp_enqueue_style( 'wp-travel-advanced-gallery-admin-trip-options-style', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-advanced-gallery-admin-trip-options.css', array( 'wp-components' ), $deps['version'] );
			} else {
				if( in_array( $screen->id, $allowed_screen ) ){
					wp_register_script( 'wp-travel-advanced-gallery', plugin_dir_url( __FILE__ ) . 'assets/js/wp-travel-advanced-gallery' . $suffix . '.js', array( 'jquery' ), '1.0.0', true );
					wp_enqueue_style( 'wp-travel-advanced-gallery', plugin_dir_url( __FILE__ ) . 'assets/css/wp-travel-advanced-gallery' . $suffix . '.css', array(), '1.0.0' );
				}
			}
		}

		/**
		 * Enqueue/Register Frontend Assets.
		 *
		 * @return void
		 */
		public static function assets_views() {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			wp_register_style(
				'wp-travel-advanced-gallery-view',
				plugin_dir_url( __FILE__ ) . 'assets/css/wp-travel-advanced-gallery-view' . $suffix . '.css',
				array( 'wp-travel-slick', 'wp-travel-popup' ),
				'1.0.0'
			);
			wp_register_script(
				'wp-travel-advanced-gallery-view',
				plugin_dir_url( __FILE__ ) . 'assets/js/wp-travel-advanced-gallery-view' . $suffix . '.js',
				array( 'jquery', 'wp-travel-popup', 'wp-travel-slick', 'wp-travel-isotope' ),
				'1.0.0',
				true
			);
		}

		/**
		 * Updates $trip_meta array.
		 *
		 * @param array $trip_meta Trip Meta.
		 * @return array
		 */
		public static function add_meta_key_values( $trip_meta ) {
			if ( isset( $_POST['wp_travel_advanced_gallery'] ) && is_array( $_POST['wp_travel_advanced_gallery'] ) ) {
				$ag = $_POST['wp_travel_advanced_gallery'];
				if ( isset( $ag['items'] ) && is_array( $ag['items'] ) ) {
					foreach ( $ag['items'] as $i => $value ) {
						$trip_meta['wp_travel_advanced_gallery']['items'][ $value['index'] ] = $value;
					}
				}
			} else {
				$trip_meta['wp_travel_advanced_gallery']['items'] = array();
			}
			return $trip_meta;
		}

		/**
		 * Backend localize data.
		 *
		 * @param array $data Gallery localized Data.
		 */
		public static function localize_gallery_data( $data ) {
			global $post;
			// if ( ! $post ) {
			if ( ! $post || get_current_screen()->id != 'itineraries' ) {
				return;
			}
			$gallery_ids      = get_post_meta( $post->ID, 'wp_travel_itinerary_gallery_ids', true );
			$advanced_gallery = get_post_meta( $post->ID, 'wp_travel_advanced_gallery', true );

			if ( is_array( $gallery_ids ) && '' === $advanced_gallery ) {
				$advanced_gallery = array();
				foreach ( $gallery_ids as $id ) {
					if ( ! wp_get_attachment_url( $id ) ) {
						continue;
					}
					$attachment = get_post( $id );
					$media_meta = array(
						'attachmentid' => $id,
						'id'           => $id,
						'url'          => wp_get_attachment_url( $id ),
						'type'         => wp_check_filetype( wp_get_attachment_url( $id ) )['type'],
						'title'        => $attachment->post_title,
						'caption'      => $attachment->post_excerpt,
					);

					$advanced_gallery['items'][] = $media_meta;
				}
			}
			$data['gallery_data']  = $advanced_gallery;
			$data['_thumbnail_id'] = get_post_meta( $post->ID, '_thumbnail_id', true );
			return $data;
		}

		/**
		 * Frontend Gallery Data.
		 *
		 * @param array $wp_travel Localized variable.
		 * @param array $settings Default Settings.
		 * @return array
		 */
		public static function localize_gallery_data_view( $wp_travel, $settings ) {
			$settings_instance = WP_Travel_Advanced_Gallery_Settings::instance();
			$settings          = $settings_instance->get_gallery_settings();

			$wp_travel['gallery_data']['settings'] = $settings;
			return $wp_travel;
		}
	}
endif;

// echo "<pre>";
// var_dump( wptravel_get_settings() );
// die;